#' Function for linking a standard runlist code to an operation-specific code
#'
#' Function for linking a standard runlist code to an operation-specific code. Function checks for correct values, but overwrites values without warning message. 
#' @param Operation Name of operation for accessing correct database.
#' @param OpCode Name of operation code
#' @param StdCode Name of equivalent standard code. By default it is the OpCode in lower case.
#' @param DBType Type of database to send the query to. Can be 'Main' (default) or 'NodeJS' for database that feeds online viewer
#' @export
#' 
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "CMHGL"
#' 
#' OpCode <- "pink"
#' setStdRunlistCode(Operation, OpCode)
#' 
#' OpCode <- "red"
#' StdCode <- "pink"
#' setStdRunlistCode(Operation, OpCode, StdCode)
#' 
#' OpCode <- "red"
#' StdCode <- "red"
#' setStdRunlistCode(Operation, OpCode)
#' 
#' OpCode <- "red"
#' setStdRunlistCode(Operation)


setStdRunlistCode <- function(Operation, OpCode, StdCode=tolower(OpCode), DBType="Main") {

  RunCodesOp  <- getRecordsFromUUID(Operation, Schema="runlist", DBTbl="code_definitions", DBType=DBType)
  RunCodesStd <- getRecordsFromUUID(Operation, Schema="runlist", DBTbl="code_standard", DBType=DBType)
  
  if (!(OpCode %in% RunCodesOp$name)) {
    stop(paste0("OpCode value not valid. Needs to be one of the following values: ", paste(RunCodesOp$name, collapse = ", ")))
  }
  
  if (!(StdCode %in% RunCodesStd$name)) {
    stop(paste0("StdCode value not valid. Needs to be one of the following values: ", paste(RunCodesStd$name, collapse = ", ")))
  }
  
  Query <- paste0("UPDATE runlist.code_definitions SET codestandard_uuid = '", RunCodesStd$uuid[RunCodesStd$name==StdCode], "' WHERE name='", OpCode, "'")
  sendQueryToGPSDB(Operation, Query, DBType=DBType)

}