#' Determines closest skilines and ski runs and updates link in database in gps.runs table. 
#'
#' Determines closest skilines and ski runs and updates link in database in gps.runs table. If these are multipe run/lines that are closest, the first one is taken. CAUTION: THIS FUNCTION OVERWRITES ALL EXISTING RECORDS IN DB!
#' @param Operation Name of operation.
#' @param GpsUnit Name of GPS unit.
#' @param DateStart Starting date of date range.
#' @param DateEnd End date of date range (equal to DateStart by default).
#' @param SkiLinesMaxDist Maximum average distance between start/end point of track and ski line for the ski line to be considered for naming the track.
#' @param SkiRunMinDist Minimum length of GPS track within a ski run polygon for the ski run to be considered for naming the track.
#' @param DBType Operational parameter for specifying from which database the run uuids should be retrieved. Values can be 'Main' (default) or 'NodeJS'.
#' @param QCOnly Switch for only including quality controlled runs.
#' @param Verbose Switch for printing of query
#' @param UserConfirm Switch whether user needs to confirm before records in database are overwritten. Default is T.
#' 
#' @examples 
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Operation <- "MPS"
#' 
#' updateGPSRuns2LoccatLink(Operation, DateStart = "2015-10-01", DateEnd = "2016-05-01")
#' 
#' @export

updateGPSRuns2LoccatLink <- function(Operation, GpsUnit=NULL, DateStart, DateEnd=DateStart, SkiLinesMaxDist=500, SkiRunMinDist=500, DBType="Main", QCOnly=T, Verbose=F, ShowProgress=T, UserConfirm=T) {

  ## Gets location catalog
  print("Getting location catalog information ...")
  
  Query <- "SELECT Count(loccat.skilines.uuid) as count FROM loccat.skilines where flag_delete = 'No'"
  SkiLinesCount <- getRecordsFromQuery(Operation=Operation, Query=Query, DBType=DBType, Verbose=Verbose)$count
  if(SkiLinesCount>0) {print(paste0("-> assigning ski line links (", SkiLinesCount, " ski lines)"))}
  
  Query <- "SELECT Count(loccat.skiruns.uuid) as count FROM loccat.skiruns where flag_delete = 'No'"
  SkiRunsCount <- getRecordsFromQuery(Operation=Operation, Query=Query, DBType=DBType, Verbose=Verbose)$count
  if(SkiRunsCount>0) {print(paste0("-> assigning ski run links (", SkiRunsCount, " ski runs)"))}
    

  ## Rectrieves GPS runs
  print("Retrieving GPS runs ...")
  GPSRuns_UUID <- getRunUUIDsFromTrackerUnitAndDate(Operation, GpsUnit=GpsUnit, DateStart=DateStart, DateEnd=DateEnd)
  
  ## Progress interval
  if(length(GPSRuns_UUID)>2000) {
    CounterInterval <- 1000
  } else if(length(GPSRuns_UUID)>1000) {
    CounterInterval <- 500
  } else if(length(GPSRuns_UUID)>500) {
    CounterInterval <- 200
  } else if(length(GPSRuns_UUID)>100) {
    CounterInterval <- 100
  } else if(length(GPSRuns_UUID)>50) {
    CounterInterval <- 10
  } else if(length(GPSRuns_UUID)>20) {
    CounterInterval <- 5
  } else {
    CounterInterval <- 1
  }
  
  ## Output DF
  OutputDF <- data.frame(uuid=GPSRuns_UUID,
                         loccatskilines_uuid=NA,
                         loccatskilines_dist=NA,
                         loccatskilines_multiple=NA,
                         loccatskiruns_uuid=NA,
                         loccatskiruns_dist=NA,
                         loccatskiruns_multiple=NA)
  
  
  ## Plotting for testing
  # plot(LocCat)
  # plot(GPSRuns$Runs, col="red", lwd=2, add=T)
  
  ## LOOPING THROUGH GPS RUNS
  for (IndexGPSRuns in 1:nrow(OutputDF)) {

    ## User feedback
    if (ShowProgress) {
      if (IndexGPSRuns==nrow(OutputDF)) {
        print(paste0("   Processing GPS run ", IndexGPSRuns, " of ", nrow(OutputDF), "."))
      } else if ((IndexGPSRuns==1) | (IndexGPSRuns %% CounterInterval == 0)) {
        print(paste0("   Processing GPS run ", IndexGPSRuns, " of ", nrow(OutputDF), " ..."))
      }
    }
    
    ## Set flag
    FoundSkiRun <- FALSE
    
    ## LENGTH OF DISTANCE WITHIN POLYGON
    if(SkiRunsCount>0) {
      
      ## Query
      Query <- paste0("SELECT gpsruns_uuid, loccatskiruns_uuid, ", 
                      "round(ST_length(ST_Transform(TblIntersect.geom, (SELECT srdi FROM misc.local_srdi LIMIT 1)))) AS length_within_m ",
                      "FROM ",
                      "(SELECT loccat.skiruns.uuid AS loccatskiruns_uuid, gps.runs.uuid AS gpsruns_uuid, ",
                      "ST_Intersection (gps.runs.geom, loccat.skiruns.geom) AS geom ",
                      "FROM gps.runs, loccat.skiruns ",
                      "WHERE gps.runs.uuid = '", OutputDF$uuid[IndexGPSRuns], "' AND loccat.skiruns.flag_delete='No') AS TblIntersect ",
                      "WHERE ST_AsText (TblIntersect.geom) != 'GEOMETRYCOLLECTION EMPTY' ",
                      "ORDER BY length_within_m DESC ")
      
      Results <- getRecordsFromQuery(Operation=Operation, Query=Query, DBType=DBType, Verbose=Verbose)
      Results <- Results[Results$length_within_m>=SkiRunMinDist,]
      
      
      if(nrow(Results)==1) {
        
        OutputDF$loccatskiruns_uuid[IndexGPSRuns] <- Results$loccatskiruns_uuid[1]
        OutputDF$loccatskiruns_dist[IndexGPSRuns] <- Results$length_within_m[1]
        OutputDF$loccatskiruns_multiple[IndexGPSRuns] <- FALSE
        
        FoundSkiRun <- TRUE
        
      } else if (nrow(Results)>1) {
        
        OutputDF$loccatskiruns_uuid[IndexGPSRuns] <- Results$loccatskiruns_uuid[1]
        OutputDF$loccatskiruns_dist[IndexGPSRuns] <- Results$length_within_m[1]
        OutputDF$loccatskiruns_multiple[IndexGPSRuns] <- TRUE
        
        FoundSkiRun <- TRUE
        
      } else {
        
        ## do nothing!
        
      }
      
    }
    
    ## AVERAGE DISTANCE OF START AND END POINT TO SKILINE
    if(SkiLinesCount>0 & !FoundSkiRun) {

      ## Query
      Query <- paste0("SELECT * ", 
                      "FROM ( ", 
                            "SELECT Tbl_Start.uuid, ", 
                                   "Tbl_Start.distance as distance_start, ", 
                                   "Tbl_End.distance as distance_end, ", 
                                   "(Tbl_Start.distance + Tbl_End.distance)/2 as distance_avg ", 
                            "FROM (", 
                                  "SELECT loccat.skilines.uuid, ", 
                                         "ST_Distance((SELECT ST_Transform(ST_StartPoint(gps.runs.geom), (SELECT srdi FROM misc.local_srdi LIMIT 1)) ", 
                                  "FROM gps.runs ", 
                                  "WHERE gps.runs.uuid = '", OutputDF$uuid[IndexGPSRuns], "' LIMIT 1) , ST_Transform(geom, (SELECT srdi FROM misc.local_srdi LIMIT 1))) as distance ", 
                                  "FROM loccat.skilines ",
                                  "WHERE loccat.skilines.flag_delete='No' ",
                                  "ORDER BY distance ASC ", 
                                  "limit 10) AS Tbl_Start ", 
                            "INNER JOIN (", 
                                  "SELECT loccat.skilines.uuid, ", 
                                         "ST_Distance((SELECT ST_Transform(ST_EndPoint(gps.runs.geom), (SELECT srdi FROM misc.local_srdi LIMIT 1)) ", 
                                  "FROM gps.runs ", 
                                  "WHERE gps.runs.uuid = '", OutputDF$uuid[IndexGPSRuns], "' LIMIT 1) , ST_Transform(geom, (SELECT srdi FROM misc.local_srdi LIMIT 1))) as distance ", 
                                  "FROM loccat.skilines ", 
                                  "WHERE loccat.skilines.flag_delete='No' ",
                                  "ORDER BY distance ASC ", 
                                  "limit 10) AS Tbl_End ", 
                             "ON (Tbl_End.uuid=Tbl_Start.uuid) ", 
                             "ORDER BY distance_avg) AS Tbl ", 
                      "WHERE distance_avg < ", SkiLinesMaxDist, ";") 
      
      Results <- getRecordsFromQuery(Operation=Operation, Query=Query, DBType=DBType, Verbose=Verbose)
      
      if(nrow(Results)==1) {
        
        OutputDF$loccatskilines_uuid[IndexGPSRuns] <- Results$uuid[1]
        OutputDF$loccatskilines_dist[IndexGPSRuns] <- round(Results$distance_avg[1])
        OutputDF$loccatskilines_multiple[IndexGPSRuns] <- FALSE
        
      } else if (nrow(Results)>1) {
        
        OutputDF$loccatskilines_uuid[IndexGPSRuns] <- Results$uuid[1]
        OutputDF$loccatskilines_dist[IndexGPSRuns] <- round(Results$distance_avg[1])
        OutputDF$loccatskilines_multiple[IndexGPSRuns] <- TRUE
        
        
      } else {
        
        ## do nothing!
        
      }

    }
      
  }

  ## UPDATING DB TABLE IN RESPONSE TO USER CONFIRMATION
  if (UserConfirm==F) {
    UserResponse <- "Yes"
  } else {
    UserResponse <- readline("Type 'Yes' to confirm overwriting of link records in DB: ")
  }
  
  if (UserResponse=="Yes") {
    
    ## Writing into DB
    if (ShowProgress) {
      updateTblInGPSDB(OutputDF, Operation, "gps", "runs", WriteToHistTbl=F, ShowCounter=CounterInterval)
    } else {
      updateTblInGPSDB(OutputDF, Operation, "gps", "runs", WriteToHistTbl=F, ShowCounter=0)
    }
    
  }
    
}
