#' Update NodeJS database with latest information.
#'
#' Update NodeJS database with latest information for specified Operation. 
#' @param Operation Name of operation
#' @param RunlistUpdate Boolean flag whether runlist information should be updated.
#' @param RunlistDateStart Start date for run list information to be transferred 
#' @param RunlistDateEnd End date for run list information to be transferred
#' @param LocCatFlagDelete Value of 'flag_delete' for updating location catalog on NodeJS server. Default is 'No'
#' @param Verbose Switch for printing SQL query. Default value is FALSE.
#' @param MaxNumTries Number of maximum tries that are attempted. Default value is 10. 
#' @export

updateNodeJS <- function(Operation, RunlistUpdate=T, RunlistDateStart=NA, RunlistDateEnd=RunlistDateStart, LocCatFlagDelete='No', Verbose=FALSE, MaxNumTries=10) {

  
  ## Generic function for transfer of db tables
  ## ******************************************
  
  transferDBTbl <- function(Operation, Schema, DBTbl, DateStart=NA, DateEnd=DateStart, DateCol="date", DBType_From="Main", DBType_To="NodeJS", flag_delete=NA, MaxRecNumPerDBCall=2000) {
    
    print(paste0("*** PROCESSING ", Schema, ".", DBTbl, " ***"))
    
    if(!is.na(DateStart)) {
      Query <- paste0("SELECT uuid from ", Schema, ".", DBTbl, " WHERE ", DateCol, ">='", DateStart, "' and ", DateCol, "<='", DateEnd, "'")
      if (!(is.na(flag_delete))) {
        Query <- paste0(Query, " AND flag_delete='", flag_delete, "'")  
      }
    } else {
      Query <- paste0("SELECT uuid from ", Schema, ".", DBTbl)
      if (!(is.na(flag_delete))) {
          Query <- paste0(Query, " WHERE flag_delete='", flag_delete, "'")  
      }
    }
    
    UUID_From <- getRecordsFromQuery(Operation, Query, DBType=DBType_From)$uuid
    UUID_To   <- getRecordsFromQuery(Operation, Query, DBType=DBType_To)$uuid
    UUID_Diff <- UUID_From[!(UUID_From %in% UUID_To)]
    
    if (length(UUID_Diff)>0) {
      
      if (length(UUID_Diff) > 1000) {
        CountInterval <- 500
      } else if (length(UUID_Diff) > 100) {
        CountInterval <- 100
      } else if (length(UUID_Diff) > 10) {
        CountInterval <- 20
      } else {
        CountInterval <- 1
      }
      
      if (length(UUID_Diff) <= MaxRecNumPerDBCall) {
      
        Tbl_Diff <- getRecordsFromUUID(Operation, Schema, DBTbl, UUID=UUID_Diff, GeomCol="", SuppressOrderWarning=T)
        writeDFToGPSDB(Tbl_Diff, Operation, Schema, DBTbl, DBType=DBType_To, ObjLabel = paste(DBTbl, "record"), ShowCounter = CountInterval)
        
      } else {
        
        NumCalls <- ceiling(length(UUID_Diff)/MaxRecNumPerDBCall)
        print(paste0(length(UUID_Diff), " records to be transferred in ", NumCalls, " calls."))
        
        for (Call in 1:NumCalls) {
          
          IndexStart <- (Call-1)*MaxRecNumPerDBCall+1
          IndexEnd <- ifelse(Call<NumCalls, Call*MaxRecNumPerDBCall, length(UUID_Diff))
          print(paste0("Call ", Call, " of ", NumCalls, ": Index ", IndexStart, "-", IndexEnd))
          
          Tbl_Diff <- getRecordsFromUUID(Operation, Schema, DBTbl, UUID=UUID_Diff[IndexStart:IndexEnd], GeomCol="", SuppressOrderWarning=T)
          writeDFToGPSDB(Tbl_Diff, Operation, Schema, DBTbl, DBType=DBType_To, ObjLabel = paste(DBTbl, "record"), ShowCounter = CountInterval)
          
        }
        
      }
      
    } else {
      print("No records to be processed!")
    }
    
  }
  
  ## Wrapping function to allow use tryMultipleTimes
  updateNodeJSTry <- function(Operation, RunlistUpdate=T, RunlistDateStart=NA, RunlistDateEnd=RunlistDateStart, LocCatFlagDelete='No', Verbose=FALSE) {
  
    ## Start and end dates
    ## *******************
    
    ## Transfer of gps tables
    ## **********************
  
    transferDBTbl(Operation, "loccat", "oparea", flag_delete = LocCatFlagDelete)
    transferDBTbl(Operation, "loccat", "zones", flag_delete = LocCatFlagDelete)
    transferDBTbl(Operation, "loccat", "catroads", flag_delete = LocCatFlagDelete)
    transferDBTbl(Operation, "loccat", "skilines", flag_delete = LocCatFlagDelete)
    transferDBTbl(Operation, "loccat", "skiruns", flag_delete = LocCatFlagDelete)
    
    transferDBTbl(Operation, "gps", "tracks")
    transferDBTbl(Operation, "gps", "tracks_days")
    transferDBTbl(Operation, "gps", "runs")
    transferDBTbl(Operation, "gps", "runs_pausepoints")
    
    transferDBTbl(Operation, "gis", "gpsruns_segments_elevband")
    
    transferDBTbl(Operation, "infoex", "hzdassess")
    transferDBTbl(Operation, "infoex", "hzdassess_loc_x")
    transferDBTbl(Operation, "infoex", "hzdassess_avprob")
    
    transferDBTbl(Operation, "runlist", "code_standard")
    transferDBTbl(Operation, "runlist", "code_definitions")
    transferDBTbl(Operation, "runlist", "runlists", DateStart=RunlistDateStart, DateEnd=RunlistDateEnd)
    transferDBTbl(Operation, "runlist", "codes_catroads", DateStart=RunlistDateStart, DateEnd=RunlistDateEnd)
    transferDBTbl(Operation, "runlist", "codes_skiruns_skilines", DateStart=RunlistDateStart, DateEnd=RunlistDateEnd)
    
    
    ## Updating of public.tracker_clr table in both Main and NodeJS
    ## ************************************************************
    
    Query <- "SELECT gps.tracks.unit AS gpstracks_unit, Null AS clr  FROM gps.tracks GROUP BY gps.tracks.unit ORDER BY gps.tracks.unit ASC"
    Trackers_All <- getRecordsFromQuery(Operation, Query, DBType="NodeJS")
    Query <- "SELECT gpstracks_unit, clr FROM public.tracker_clr"
    Trackers_WithClr <- getRecordsFromQuery(Operation, Query, DBType="NodeJS")
    
    Trackers_New <- Trackers_All[!(Trackers_All$gpstracks_unit %in% Trackers_WithClr$gpstracks_unit),]
    
    if (nrow(Trackers_New)>0) {
    
      print(paste0("Added ", nrow(Trackers_New), " new trackers with colors to public.tracker_clr table."))
      
      ## Array of colors
      ## from http://phrogz.net/css/distinct-colors.html with interleave value 6
      ClrArray_All <- c("#ff8080", "#7f6240", "#29a629", "#b6def2", "#0e0033", "#cc3370", "#7f4040", "#e59900", "#00f220", "#002240", "#9900e6", "#590018", "#402020", "#f2e6b6", "#003322", "#266399", "#3d004d", "#b38692", "#bf3300", "#403d10", "#00ffcc", "#303640", "#992691", "#ff0022", "#ffd0bf", "#eeff00", "#397367", "#4073ff", "#f279da", "#8c0013", "#662900", "#838c00", "#6cd2d9", "#9999cc", "#331a2b", "#ff8c40", "#2e661a", "#268299", "#4f468c", "#664d5e", "#e5b073", "#87a67c", "#00aaff", "#2c00a6", "#661a42")
      
      ClrArray_NotUsed <- ClrArray_All[!(ClrArray_All %in% Trackers_WithClr$clr)]
      Trackers_New$clr <- ClrArray_NotUsed[1:nrow(Trackers_New)]
      
      writeDFToGPSDB(Trackers_New, Operation, "public", "tracker_clr", DBType="Main", ShowCounter = 0)
      writeDFToGPSDB(Trackers_New, Operation, "public", "tracker_clr", DBType="NodeJS", ShowCounter = 0)
      
    } else {
    
      print(paste0("No new trackers added to public.tracker_clr table."))
      
    }
    
  }
  
  ## Actual call of function
  ## ***********************
  tryMultipleTimes(updateNodeJSTry(Operation=Operation, RunlistUpdate=RunlistUpdate, RunlistDateStart=RunlistDateStart, RunlistDateEnd=RunlistDateEnd, LocCatFlagDelete=LocCatFlagDelete, Verbose=Verbose), MaxNumTries=MaxNumTries)
  
}