#' Update coordinates of raster cells along ski runs stored in Postgres/PostGIS GPS database.
#'
#' Update coordinates of raster cells along ski runs stored in Postgres/PostGIS GPS database.
#' @param Operation Name of operation
#' @param User Name of user. Required for local root folder for location of raster. Default value is 'SARPLab'.
#' @param RasterName Name of raster for extraction of coodinates. Default is Incline
#' @param BufferDist size of buffer around GPS ski run in metres. Default value array of 0, 20, 40 & 60 m.
#' @param DecimalDigitsForCoordinates Decimal digits for extracted values. Default value is 5.
#' @param OnlyQCRuns Switch to specify whether to process only runs that are quality controlled (default) or all runs. Default value is TRUE
#' @param Overwrite Switch to specify whether to erase all records from the table before (re-) processing the runs. Default value id FALSE.
#' @param UserConfirm Flag whether the user needs to actively confirm deletion. True by default.
#' @param Test Only the first 10 runs are processed in Test mode. Default value is FALSE.
#' @param Verbose Switch for printing of query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off. Default value is TRUE.
#'
#' @export

updateStoredRasterCoordGPSRuns <- function(Operation, User="SARPLab", RasterName="Incline", BufferDist=c(0, 20, 40, 60), DecimalDigitsForCoordinates=5, OnlyQCRuns=T, Overwrite=F, UserConfirm=T, Test=F, Verbose=F, SuppressPostgreSQLWarnings=T) {

  ## Defaults
  DBType <- "Main"

  ## Delete existing records if Overwirte==T
  ## ***************************************
  
  if (Overwrite) {
    
    if (UserConfirm==F) {
      UserResponse <- "Yes"
    } else {
      ExistRecords <- getRecordsFromQuery(Operation, "SELECT gpsruns_uuid FROM gis.gpsruns_array_coord;")$gpsruns_uuid
      UserResponse <- readline(paste0("Type 'Yes' to confirm deletion of ", length(ExistRecords), " existing records:"))
    }
    
    ## Deletion in response to user confirmation
    if (UserResponse=="Yes") {
      Query <- "DELETE FROM gis.gpsruns_array_coord"
      sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    }
    
    rm(ExistRecords, UserResponse)
  }
  
  
  ## Retrieve UUIDs of runs to be processed
  ## **************************************
  
  if (OnlyQCRuns) {
    Query <- "SELECT gps.qc_runs.uuid FROM gps.qc_runs LEFT JOIN gis.gpsruns_array_coord ON gis.gpsruns_array_coord.gpsruns_uuid = gps.qc_runs.uuid WHERE gis.gpsruns_array_coord.gpsruns_uuid is NULL"
  } else {
    Query <- "SELECT gps.runs.uuid FROM gps.runs LEFT JOIN gis.gpsruns_array_coord ON gis.gpsruns_array_coord.gpsruns_uuid = gps.runs.uuid WHERE gis.gpsruns_array_coord.gpsruns_uuid is NULL"
  }

  RunUUIDs <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$uuid
  
  NumRuns <- length(RunUUIDs)
  Index_Runs_Max <- ifelse(Test, min(10, NumRuns), NumRuns)
  print(paste0("Number of runs to be processed: ", Index_Runs_Max, " of ", NumRuns))
  
  if (Index_Runs_Max == 0) {
    
    print("No GPS runs to be processed!")
    
  } else {
  
    ## Get raster
    ## **********
    
    Raster <- getRaster(Operation, User, RasterName)
    
    Local_proj4text <- getLocalSRDI(Operation)$proj4text
    
    
    ## Process runs
    ## ************
    SkippedRunsDueToError <- 0
    
    for (Index_Runs in 1:Index_Runs_Max) {
      
      # User feedback
      print(paste0(Sys.time(), " - Processing Run ", Index_Runs, " of ", Index_Runs_Max, " ..."))
      
      RunUUID <- RunUUIDs[Index_Runs]
      
  #     ## Skip errors
  #     tryCatch(
  #       {
      
        ## Extract coordinate values
        CoordDF <- extractRasterCoordAlongGPSRun(Operation, RunUUID, Raster, Local_proj4text, BufferDist = BufferDist)
        
        ## Create SQL for DB
        Query <- paste0("INSERT INTO gis.gpsruns_array_coord (gpsruns_uuid, source_raster, lon, lat, buffer, dist_start, dist_end) ",
                        "VALUES (",
                        convertStringToSQL(RunUUID), ", ",
                        convertStringToSQL(RasterName), ", ",
                        convertVectorToSQL(round(CoordDF$lon, DecimalDigitsForCoordinates)), ", ",
                        convertVectorToSQL(round(CoordDF$lat, DecimalDigitsForCoordinates)), ", ",
                        convertVectorToSQL(CoordDF$buffer), ", ",
                        convertVectorToSQL(CoordDF$dist_start), ", ",
                        convertVectorToSQL(CoordDF$dist_end), ")")
        
        ## Write values to DB
        sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        
  #     },
  #     error=function(e){
  #       print(paste0("Skipped run '", RunUUID, "' because unable to connect to GPSDB!!"))
  #       SkippedRunsDueToError <- SkippedRunsDueToError + 1
  #       }
  #     )
        
    }
    
    if (SkippedRunsDueToError==0) {
      print (paste0("All ", Index_Runs_Max, " runs successfully processed!"))
    } else {
      warning(paste0(SkippedRunsDueToError, " runs skipped due to DB connection errors. Please run script again to completel processing!"), immediate.=T)
    }
    
  }
  
}    
  