#' Update coordinates of raster cells within skirun polygon from location catalog stored in Postgres/PostGIS GPS database.
#'
#' Update coordinates of raster cells within skirun polygon from location catalog stored in Postgres/PostGIS GPS database.
#' @param Operation Name of operation
#' @param User Name of user. Required for local root folder for location of raster. Default value is 'SARPLab'
#' @param RasterName Name of raster for extraction of coodinates. Default is Incline
#' @param BufferDist size of buffer around GPS ski run in metres. Default value array of -60, -40, -20 & 0 m.
#' @param DecimalDigitsForCoordinates Decimal digits for extracted values. Default value is 5.
#' @param Overwrite Switch to specify whether to erase all records from the table before (re-) processing the runs. Default value id FALSE.
#' @param UserConfirm Flag whether the user needs to actively confirm deletion. True by default.
#' @param Test Only the first 10 runs are processed in Test mode. Default value is FALSE.
#' @param Verbose Switch for printing of query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off. Default value is TRUE.
#'
#' @export

updateStoredRasterCoordLoccatSkiruns <- function(Operation, User="SARPLab", RasterName="Incline", BufferDist=c(-60, -40, -20, 0), DecimalDigitsForCoordinates=5, Overwrite=F, UserConfirm=T, Test=F, Verbose=F, SuppressPostgreSQLWarnings=T) {

  ## Defaults
  DBType <- "Main"

  ## Delete existing records if Overwirte==T
  ## ***************************************
  
  if (Overwrite) {
    
    if (UserConfirm==F) {
      UserResponse <- "Yes"
    } else {
      ExistRecords <- getRecordsFromQuery(Operation, "SELECT loccatskiruns_uuid FROM gis.loccatskiruns_array_coord;")$loccatskiruns_uuid
      UserResponse <- readline(paste0("Type 'Yes' to confirm deletion of ", length(ExistRecords), " existing records:"))
    }
    
    ## Deletion in response to user confirmation
    if (UserResponse=="Yes") {
      Query <- "DELETE FROM gis.loccatskiruns_array_coord"
      sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
    }
    
    rm(ExistRecords, UserResponse)
  }
  
  
  ## Retrieve UUIDs of runs to be processed
  ## **************************************
  
  Query <- "SELECT loccat.skiruns.uuid FROM loccat.skiruns LEFT JOIN gis.loccatskiruns_array_coord ON gis.loccatskiruns_array_coord.loccatskiruns_uuid = loccat.skiruns.uuid WHERE gis.loccatskiruns_array_coord.loccatskiruns_uuid is NULL"
  
  SkirunUUIDs <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$uuid
  
  NumRuns <- length(SkirunUUIDs)
  Index_Runs_Max <- ifelse(Test, min(10, NumRuns), NumRuns)
  print(paste0("Number of runs to be processed: ", Index_Runs_Max, " of ", NumRuns))
  
  Skiruns <- getSpatialObjectsFromUUID(Operation, "loccat", "skiruns", DBType=DBType, UUID=SkirunUUIDs)
  
  ## Get raster
  ## **********
  
  Raster <- getRaster(Operation, User, RasterName)
  
  Local_proj4text <- getLocalSRDI(Operation)$proj4text
  
  
  ## Process runs
  ## ************
  SkippedRunsDueToError <- 0
  
  for (Index_Runs in 1:Index_Runs_Max) {
    
    # User feedback
    print(paste0(Sys.time(), " - Processing Skirun ", Index_Runs, " of ", Index_Runs_Max, " ..."))
    
    Skirun <- Skiruns[Index_Runs,]
    SkirunUUID <- Skirun@polygons[[1]]@ID
    
    ## Extract coordinate values
    CoordDF <- extractRasterCoordWithinPolygon(Skirun, Raster, Local_proj4text, BufferDist = BufferDist)

    if (!is.null(CoordDF)) {
        
      ## Create SQL for DB
      Query <- paste0("INSERT INTO gis.loccatskiruns_array_coord (loccatskiruns_uuid, source_raster, lon, lat, buffer) ",
                      "VALUES (",
                      convertStringToSQL(SkirunUUID), ", ",
                      convertStringToSQL(RasterName), ", ",
                      convertVectorToSQL(round(CoordDF$lon, DecimalDigitsForCoordinates)), ", ",
                      convertVectorToSQL(round(CoordDF$lat, DecimalDigitsForCoordinates)), ", ",
                      convertVectorToSQL(CoordDF$buffer), ")")
      
      ## Write values to DB
      sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

    } else {
      
      print(paste0("Skipped run '", SkirunUUID, "' because of empty CoordDF!"))
       SkippedRunsDueToError <- SkippedRunsDueToError + 1
      
    }
            
  }
  
  if (SkippedRunsDueToError==0) {
    print (paste0("All ", Index_Runs_Max, " runs successfully processed!"))
  } else {
    warning(paste0(SkippedRunsDueToError, " runs skipped due to DB connection errors. Please run script again to completel processing!"), immediate.=T)
  }
  
}    
  