#' Extract values from raster within location catalog ski runs
#'
#' Extract values from raster within location catalog ski runs and stores them in PostgreSQL/PostGIS GPS database.
#' @param Operation Name of operation
#' @param User Name of user. Required for local root folder for location of raster. Default value is 'SARPLab'.
#' @param RasterNames Name of raster(s) to be processed. Can be single string or array of strings.
#' @param Method Method for extraction of raster values. Must be 'simple' or 'bilinear'. See help for raster::extract for more detail.
#' @param DecimalDigits Decimal digits for extracted values. Default value is 0 (zero). Can be single number of array of numbers. If an array, it needs to be os same size as RasterNames.
#' @param Overwrite Switch to specify whether to erase all records from the table before (re-) processing the runs. Default value id FALSE.
#' @param Test Only the first 10 runs are processed in Test mode. Default value is FALSE.
#' @param UserConfirm Flag whether the user needs to actively confirm deletion. True by default.
#' @param Verbose Switch for printing of query. Default value is FALSE.
#' @param SuppressPostgreSQLWarnings Switch for turning warnings off. Default value is TRUE.
#'
#' @export

updateStoredRasterValuesLoccatSkiruns <- function(Operation, RasterNames, User="SARPLab", Method="simple", DecimalDigits=0, Overwrite=F, Test=F, UserConfirm=T, Verbose=F, SuppressPostgreSQLWarnings=T) {

  ## Default
  DBType <- "Main"

  ## Validate input parameters
  ## =========================

  ## Lenth of RasterNames and DecimalDigits arrays
  if (length(RasterNames)>1) {
    if (length(DecimalDigits)==1) {
      ## ok
    } else if (length(DecimalDigits)==length(RasterNames)) {
      ## ok
    } else {
      stop("The parameter DecimalDigits needs to be either a single value or an array of the same length as RasterNames!")
    }
  }

  ## Methods
  if (Method!="simple" & Method!="bilinear") {
    stop("Value of Method parameter must be 'simple' or 'bilinear'! See help for raster::extract for more detail.")
  }

  ## Get raster background information from DB
  RasterInfoAll <- listRasterRefAndArrayTbl(Operation)

  ## Loop for rasternames
  ## ====================
  
  NumToBeProcessed <- Counter <- rep(0, length(RasterNames))
  
  for (Index_Raster in 1:length(RasterNames)) {

    RasterInfo <- subset(RasterInfoAll, name==RasterNames[Index_Raster])

    if (nrow(RasterInfo)==1) {

      ## Get basic information and raster
      ## *******************************
      RasterName <- RasterInfo$name
      DBTbl <- paste0("loccatskiruns_array_", RasterInfo$gis_array_name_ext)

      DecimalDigitsForRaster <- ifelse(length(DecimalDigits)>1, DecimalDigits[Index_Raster], DecimalDigits)

      print(paste0("Getting raster '", RasterName, "' ..."))
      Raster <- getRaster(Operation, User, RasterName)


      ## Delete existing records if Overwirte==T
      ## ***************************************
      if (Overwrite) {

        if (UserConfirm==F) {
          UserResponse <- "Yes"
        } else {
          Query <- paste0("SELECT loccatskiruns_uuid FROM gis.", DBTbl, ";")
          ExistRecords <- getRecordsFromQuery(Operation, Query)$loccatskiruns_uuid
          UserResponse <- readline(paste0("Type 'Yes' to confirm deletion of ", length(ExistRecords), " existing records:"))
        }

        ## Deletion in response to user confirmation
        if (UserResponse=="Yes") {
          Query <- paste0("DELETE FROM gis.", DBTbl)
          sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
        }
      }

      ## Retrieve UUIDs of runs to be processed
      ## **************************************
      Query <- paste0("SELECT gis.loccatskiruns_array_coord.loccatskiruns_uuid FROM gis.loccatskiruns_array_coord LEFT JOIN gis.", DBTbl, " ON gis.", DBTbl, ".loccatskiruns_uuid = gis.loccatskiruns_array_coord.loccatskiruns_uuid WHERE gis.", DBTbl, ".loccatskiruns_uuid is NULL")
      
      RunUUIDs <- getRecordsFromQuery(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)$loccatskiruns_uuid

      NumRuns <- length(RunUUIDs)
      Index_Runs_Max <- ifelse(Test, min(10, NumRuns), NumRuns)
      
      NumToBeProcessed[Index_Raster] <- Index_Runs_Max
      print(paste0("Number of runs to be processed: ", Index_Runs_Max, " of ", NumRuns))

      ## Processing GPS runs
      ## ===================
      if (Index_Runs_Max == 0) {

        print(paste0("No loccat skiruns to be processed for ", RasterName, "!"))

      } else {

        ## Loop for GPS runs
        for (Index_Runs in 1:Index_Runs_Max) {
          
#           ## Skip errors
#           tryCatch({
              
            RunUUID <- RunUUIDs[Index_Runs]

            # User feedback
            print(paste0(Sys.time(), " ", RasterName, " - Processing Run ", Index_Runs, " of ", Index_Runs_Max, " ..."))

            # Get Raster Coordinate for ski line
            CoordDF <- data.frame(lat = convertArrayFromSQL(getRecordsFromUUID(Operation, "gis", "loccatskiruns_array_coord", UUID=RunUUID, UUIDCol="loccatskiruns_uuid", ResultCol="lat", Verbose=F, SuppressPostgreSQLWarnings=T, SuppressOrderWarning=T)$lat),
                                  lon = convertArrayFromSQL(getRecordsFromUUID(Operation, "gis", "loccatskiruns_array_coord", UUID=RunUUID, UUIDCol="loccatskiruns_uuid", ResultCol="lon", Verbose=F, SuppressPostgreSQLWarnings=T, SuppressOrderWarning=T)$lon))
            
            # Extract values
            print("Extracting values from raster ...")
            ValueArray <- round(extractRasterValuesFromCoord(Raster, CoordDF, Method), DecimalDigitsForRaster)
            
            ## Create SQL for DB
            Query <- paste0("INSERT INTO gis.", DBTbl, " (loccatskiruns_uuid, valuearray) ",
                            "VALUES (", convertStringToSQL(RunUUID), ", ",
                            convertVectorToSQL(ValueArray), ")")
           
            ## Write values to DB
            sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)
            
            ## Updating Counter
            Counter[Index_Raster] <- Counter[Index_Raster]+1
            
#          },
#          error=function(e){
#            print(paste0("Skipped run '", RunUUID, "' because unable to connect to GPSDB!!"))
#          }
#        )

        } ## end of run loop

      } ## end of if condition: Index_Runs_Max > 0

    } else {

      warning(paste0("No raster reference available for '", RasterNames[Index_Raster], "'!"))

    }

  }
  
  for (Index_Raster in 1:length(RasterNames)) {
  
    if (NumToBeProcessed[Index_Raster]==0) {
      print (paste0(RasterNames[Index_Raster], " - No runs needed to be processed!"))
    } else if (Counter[Index_Raster]==NumToBeProcessed[Index_Raster]) {
      print (paste0(RasterNames[Index_Raster], " - All ", NumToBeProcessed[Index_Raster], " runs processed successfully!"))
    } else {
      warning(paste0(RasterNames[Index_Raster], " - ", Counter[Index_Raster]-NumToBeProcessed[Index_Raster], " runs skipped due to DB connection errors. Please run script again to completel processing!"), immediate.=T)
    }
      
  }

}

