#' Write spatial dataframe to prexisting table in PostgreSQL/PostGIS GPS database
#'
#' Writes entire content of spatial dataframe to prexisting table in PostgreSQL/PostGIS GPS database. All of the columns of the spatial dataframe need to exist in the database table.
#' @param DF Dataframe
#' @param Operation Name of operation
#' @param Schema Name of database schema that contains database table
#' @param DBTbl Name of database table with geometry
#' @param DBType Type of database: "Main" (default) or "NodeJS".
#' @param DBTblHist Name of database table that stores history information
#' @param WriteToHistTbl Boolean flag for whether the original record should first be written to a history table.
#' @param Verbose Switch for printing of query
#' @param SuppressPostgreSQLWarnings Switch for supressing warning messages from postgresqlExecStatement. Default value is TRUE. Turn on for debugging!
#' @param ShowCounter Numeric value that specifies whether counter for queries should be shown (>0) and at what interval
#' @param ObjLabel Label for making process counter more meaningful.
#' @examples 
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' 
#' Oper <- "NEH"
#' 
#' ## Retrieving existing records
#' Tbl_Flying <- getRecordsFromUUID(Oper, "obs", "cond_flying")
#' 
#' ## Modifying set of existing records
#' Tbl_FlyingToEdit <- Tbl_Flying[c(1:10), c("uuid", "comment")]
#' Tbl_FlyingToEdit$comment <- paste("TEST", c(1:10))
#' 
#' ## Updating modified records in database
#' updateTblInGPSDB(Tbl_FlyingToEdit, Oper, "obs", "cond_flying")
#' 
#' ## Checking updated records
#' Tbl_Flying_Updated <- getRecordsFromUUID(Oper, "obs", "cond_flying", UUID=Tbl_FlyingToEdit$uuid)
#' 
#' ## Checking complete history on updated records
#' Tbl_Flying_History <- getRecordsFromUUID(Oper, "obs", "cond_flying_h", UUID=Tbl_FlyingToEdit$uuid, AddRowNames=F)
#' 
#' @export

updateTblInGPSDB <- function(DF, Operation, Schema, DBTbl, DBType="Main", DBTblHist=paste0(DBTbl, "_h"), WriteToHistTbl=T, Verbose=F, SuppressPostgreSQLWarnings=T, ShowCounter=1, ObjLabel="row") {

  ## Check for UUID column
  if (!("uuid" %in% names(DF))) {stop("Dataframe is missing 'uuid' column!")}
  
  ## UUID column as first column
  refcols <- c("uuid")
  DF <- DF[, c(refcols, setdiff(names(DF), refcols))]
  
  
  ## Writing original records to history table
  if (WriteToHistTbl) {
    
    print("Writing original records to history table ...")

    ## Get original records
    OrigRecords <- getRecordsFromUUID(Operation=Operation,
                                      Schema=Schema, DBTbl=DBTbl, DBType=DBType,
                                      UUID=DF$uuid,
                                      Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings, SuppressOrderWarning=T)
    
    ## Write original records into history table
    ShowCounterHistory <- 0
    writeDFToGPSDB(DF=OrigRecords,
                   Operation=Operation,
                   Schema=Schema, DBTbl=DBTblHist, DBType=DBType,
                   Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings, ShowCounter=ShowCounterHistory)
    
  }

  ## Establish DB connection
  DBCon <- connectToGPSDB(Operation, ReadOnly=F, DBType=DBType)

  ## Write entire DF to DB
  print("Writing updated rows to database ...")
  for (Index_Rows in 1:nrow(DF)) {

    ## User feedback
    if (ShowCounter>0) {
      if (Index_Rows==nrow(DF)) {
        print(paste0("Writing ", ObjLabel, " ", Index_Rows, " of ", nrow(DF), " to GPS DB."))
      } else if ((Index_Rows==1) | (Index_Rows %% ShowCounter == 0)) {
        print(paste0("Writing ", ObjLabel, " ", Index_Rows, " of ", nrow(DF), " to GPS DB ..."))
      }
    }

    ## Build query - start
    if (is.na(Schema)) {
      Query <- paste0("UPDATE \"", DBTbl, "\" SET ")
    } else {
      Query <- paste0("UPDATE \"", Schema, "\".\"", DBTbl, "\" SET ")
    }
    
    ## Build query - add update conditions
    for (Index_Col in 1:ncol(DF)) {
      
      if(names(DF)[Index_Col]!="uuid") {
        
        Value <- DF[Index_Rows, Index_Col]

        if(class(Value)[1]=="numeric") {
          Query <- paste0(Query, "\"", names(DF)[Index_Col], "\"", "=", convertNumericToSQL(Value))
        } else if(class(Value)[1]=="chron") {
          Query <- paste0(Query, "\"", names(DF)[Index_Col], "\"", "=", convertDateTimeToSQL(Value))
        } else if(class(Value)[1]=="dates") {
          Query <- paste0(Query, "\"", names(DF)[Index_Col], "\"", "=", convertDateToSQL(Value))
        } else if(class(Value)[1]=="logical") {
          Query <- paste0(Query, "\"", names(DF)[Index_Col], "\"", "=", convertBooleanToSQL(Value))
        } else {
          Query <- paste0(Query, "\"", names(DF)[Index_Col], "\"", "=", convertStringToSQL(Value))
        }
  
        if (Index_Col<ncol(DF)) {
          Query <- paste0(Query, ", ")
        } else {
          Query <- paste0(Query, " ")
        }

      }
      
    }
    
    ## Building query - Adding uuid
    Query <- paste0(Query, "WHERE uuid=", convertStringToSQL(DF$uuid[Index_Rows]))

    ## Send query to DB
    # sendQueryToGPSDB(Operation, Query, Verbose, SuppressPostgreSQLWarnings)
    # sendQueryToGPSDB function is too slow, because the DB connection needs to be established every time.

    if(Verbose) {cat(Query, "\n")}
    if (SuppressPostgreSQLWarnings) {options(warn=-1)}

    Result <- dbSendQuery(DBCon, Query)

    if (SuppressPostgreSQLWarnings) {options(warn=0)}

  }

  ## Close DB connection
  dbDisconnect(DBCon)
  rm(DBCon)

}
