#' Write spatial dataframe to prexisting table in PostgreSQL/PostGIS GPS database.
#'
#' Writes entire content of spatial dataframe to prexisting table in PostgreSQL/PostGIS GPS database. All of the columns of the spatial dataframe need to exist in the database table.
#' @param SPDF Spatial dataframe
#' @param Operation Name of operation
#' @param Schema Name of database schema that contains database table
#' @param DBTbl Name of database table with geometry
#' @param DBType Type of database the table should be written to. Can be 'Main' (default) or 'NodeJS' for database that feeds online viewer.
#' @param GeomCol Name of geometry column in database table
#' @param SRID Numeric reference to coordinate system for geospatial information
#' @param Overwrite Boolean flag for whether the entire table should be overwritten (i.e., all records deleted first)
#' @param Polygon2Multipolygon Boolean flag for indicating whether a polygon layer needs to be converted into a multipolygon layer
#' @param Verbose Switch for printing of query
#' @param SuppressPostgreSQLWarnings Switch for supressing warning messages from postgresqlExecStatement. Default value is TRUE. Turn on for debugging!
#' @param ShowCounter Numeric value that specifies whether counter for queries should be shown (>0) and at what interval
#' @param ObjLabel Label for making process counter more meaningful.
#' @param MaxNumTries Number of maximum tries that are attempted. Default value is 10.
#' @export

writeSPDFToGPSDB <- function(SPDF, Operation, Schema, DBTbl, DBType="Main", GeomCol="geom", SRID=4617, Overwrite=F, Polygon2Multipolygon=F, Verbose=F, SuppressPostgreSQLWarnings=T, ShowCounter=1, ObjLabel="row", MaxNumTries=10) {

  ## Extract pure DF
  DF <- SPDF@data
  ColNames <- colnames(DF)

  ## Create geometry column with specified name (requires rgeos)
  DF$geom <- rgeos::writeWKT(SPDF, byid = T)
  # Does not work for multi* geometries --> creates array instead of single string
  
  ## Convert polygon to multipolygon if necessary
  if (Polygon2Multipolygon) {
    for (i in 1:nrow(DF)) {
      if (substr(DF$geom[i], 1, 7) == "POLYGON") {
        DF$geom[i] <- paste0("MULTIPOLYGON (", substring(DF$geom[i], 9), ")")
      } else if (substr(DF$geom[i], 1, 7) == "MULTIPO") {
        # Do nothing
      } else {
        stop("Cannot apply Polygon2Multipolygon since original geometry is not polygon or multipolygon!")
      }
    }
  }
  
  ## Update name of geometry column
  colnames(DF) <- c(ColNames, GeomCol)

  ## Delete all records in destination table if requested
  if (Overwrite) {
    if (Schema=="") {
      Query <- paste0("DELETE FROM \"", DBTbl, "\"")
    } else {
      Query <- paste0("DELETE FROM \"", Schema, "\".\"", DBTbl, "\"")
    }
    if (Verbose) {
      print("Deleting existing records in GPS DB ...")
      cat(paste0(Query, "\n\n"))
    }

    ## Send query to DB
    sendQueryToGPSDB(Operation, Query, DBType=DBType, Verbose=Verbose, SuppressPostgreSQLWarnings=SuppressPostgreSQLWarnings)

  }

  ## Establish DB connection
  # DBCon <- connectToGPSDB(Operation, DBType=DBType, ReadOnly=F)
  DBCon <- tryMultipleTimes(connectToGPSDB(Operation, DBType=DBType, ReadOnly=F), MaxNumTries=MaxNumTries)

  ## Write entire DF to DB
  for (Index_Rows in 1:nrow(DF)) {

    ## User feedback
    if (ShowCounter>0) {
      if (Index_Rows==nrow(DF)) {
        print(paste0("Writing ", ObjLabel, " ", Index_Rows, " of ", nrow(DF), " to GPS DB."))
      } else if ((Index_Rows==1) | (Index_Rows %% ShowCounter == 0)) {
        print(paste0("Writing ", ObjLabel, " ", Index_Rows, " of ", nrow(DF), " to GPS DB ..."))
      }
    }

    ## Build query - start
    if (is.na(Schema)) {
      Query <- paste0("INSERT INTO \"", DBTbl, "\" (")
    } else {
      Query <- paste0("INSERT INTO \"", Schema, "\".\"", DBTbl, "\" (\"")
    }
    Query <- paste0(Query, paste(colnames(DF), collapse ="\", \""), "\") VALUES (")

    ## Build query - add values and finish query
    for (Index_Col in 1:ncol(DF)) {

      Value <- DF[Index_Rows, Index_Col]

      if (colnames(DF)[Index_Col]==GeomCol) {
        Query <- paste0(Query, "ST_GeomFromText('", Value, "', ", SRID, ")")
      } else if(class(Value)[1]=="numeric") {
        Query <- paste0(Query, convertNumericToSQL(Value))
      } else if(class(Value)[1]=="chron") {
        Query <- paste0(Query, convertDateTimeToSQL(Value))
      } else if(class(Value)[1]=="dates") {
        Query <- paste0(Query, convertDateToSQL(Value))
      } else {
        Query <- paste0(Query, convertStringToSQL(Value))
      }

      if (Index_Col<ncol(DF)) {
        Query <- paste0(Query, ", ")
      } else {
        Query <- paste0(Query, ")")
      }

    }

    ## Send query to DB
    # sendQueryToGPSDB(Operation, Query, DBType, Verbose, SuppressPostgreSQLWarnings)
    # sendQueryToGPSDB function is too slow, becasue the DB connection needs to be established every time.

    if(Verbose) {cat(Query, "\n")}
    if (SuppressPostgreSQLWarnings) {options(warn=-1)}

    Result <- dbSendQuery(DBCon, Query)

    if (SuppressPostgreSQLWarnings) {options(warn=0)}

  }

  ## Close DB connection
  dbDisconnect(DBCon)
  rm(DBCon)

}
