#' Function for plotting hazard chart with Av problems.
#'
#' Function for plotting hazard chart with Av problems.
#' @param HzdAssessObj InfoEx hazard assessment object.
#' @param ElevBand Switch for specifying elevation band. Default is 'All'. Additional possible values are 'Alp', 'Tl' or 'Btl'.
#' @param AvObs Optional dataframe with avalanche observations (default=NULL). IMPLEMENTATION NEED UPDATING!
#' @param SHObsDates Optional array with dates when SH was observed as AM/PM surface forms (default=NULL).
#' @param WithXAxis Switch for plotting x-axis (default=TRUE).
#' @param DateStart Start date for plot. If not provided, the function takes the date range of TblWx (default=NA).
#' @param DateEnd End date for plot. If not provided, the function takes the date range of TblWx (default=NA).
#' @param HighlightDateStart Start of highlighted time period (default=NA).
#' @param HighlightDateEnd End of highlighted time period (default=HighlightDateStart).
#' @param HighlightCol Color for highlighted time period (default="light grey").
#' @param cex.axis cex.axis (default=1.0).
#' @param cex.lab cex.lab (default=1.0).
#' @param cex.points cex.points (default=1.6).
#' @param plotElevLabels Switch for plotting the elevation band labels (default=F).
#' @param plotAvProb Switch for plotting Av problems (default=TRUE).
#'
#' @examples
#' require(SarpGeneralVis)
#'
#' ## InfoEx avalanche hazard objects
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#'
#' Operation <- "MPS"
#'
#' DateStart <- "2015-12-15"
#' DateEnd <- "2016-04-15"
#'
#' ## Get HzdAssess object for PM assessment
#' HzdAssess <- getInfoExHzdAssessObject(Operation, DateStart=DateStart, DateEnd=DateEnd, Time="PM")
#'
#' ## Standard plot
#' plotTSHzdRatingAvProb(HzdAssess, main="Hazard assessment for MPS (Winter 2016)")
#'
#' ## Bulletin avalanche hazard objects
#' require(SarpBulletinTools)
#'
#' load(url("http://data.avalancheresearch.ca/AllBulletins_2010To2018.RData"))
#' GlacierBullID <- Bulletins$Bulletins$BULLETIN_ID[Bulletins$Bulletins$REGION=="Glacier" & Bulletins$Bulletins$SEASON==2017]
#' GlacierBull <- extractFromBulletinObj(Bulletins, ByBulletinID = GlacierBullID)
#' summary(GlacierBull)
#'
#' plotTSHzdRatingAvProb(GlacierBull, main="Glacier NP avalanche problems (winter 2017)")
#' plotTSHzdRatingAvProb(GlacierBull, ElevBand = "alp", main="Glacier NP avalanche problems (winter 2017) - ALP only")
#' plotTSHzdRatingAvProb(GlacierBull, ElevBand = "tl", main="Glacier NP avalanche problems (winter 2017) - TL only")
#' plotTSHzdRatingAvProb(GlacierBull, ElevBand = "btl", main="Glacier NP avalanche problems (winter 2017) - BTL only")
#'
#' @export



plotTSHzdRatingAvProb <- function(HzdAssessObj,
                              ElevBand = "All",
                                AvObs=NULL,
                                SHObsDates=NULL,
                                WithXAxis=T,
                                DateStart=NA,
                                DateEnd=NA,
                                main="",
                                HighlightDateStart=NA,
                                HighlightDateEnd=HighlightDateStart,
                                HighlightCol="light grey",
                                NoGlideNoCornice=F,
                                cex.axis=1.0,
                                cex.lab=1.0,
                                cex.points=1.6,
                                plotElevLabels=F,
                                plotAvProb=T) {

  print("... starting plotTSHzdRatingAvProb")

  ## Check
  ElevBand <- toupper(ElevBand)
  if (ElevBand!="ALL" & ElevBand!="ALP" & ElevBand!="TL" & ElevBand!="BTL") {
    stop("Value of ElevBand parameter must be, 'All', 'Alp', 'tl' or 'Btl'!'")
  }

  ## Conversion
  if (names(HzdAssessObj)[1]=="HzdAssess") {

    ## Creating of TblPlot
    TblPlot <- data.frame(id = HzdAssessObj$HzdAssess$uuid,
                          date = HzdAssessObj$HzdAssess$date_assess,
                          dng_alp = NA, dng_tl = NA, dng_btl = NA,
                          avpro_ldry = 0,  avpro_ldry_alp = 0,  avpro_ldry_tl = 0,  avpro_ldry_btl = 0,
                          avpro_lwet = 0,  avpro_lwet_alp = 0,  avpro_lwet_tl = 0,  avpro_lwet_btl = 0,
                          avpro_wets = 0,  avpro_wets_alp = 0,  avpro_wets_tl = 0,  avpro_wets_btl = 0,
                          avpro_corn = 0,  avpro_corn_alp = 0,  avpro_corn_tl = 0,  avpro_corn_btl = 0,
                          avpro_wind = 0,  avpro_wind_alp = 0,  avpro_wind_tl = 0,  avpro_wind_btl = 0,
                          avpro_storm = 0, avpro_storm_alp = 0, avpro_storm_tl = 0, avpro_storm_btl = 0,
                          avpro_pers = 0,  avpro_pers_alp = 0,  avpro_pers_tl = 0,  avpro_pers_btl = 0,
                          avpro_dpers = 0, avpro_dpers_alp = 0, avpro_dpers_tl = 0, avpro_dpers_btl = 0,
                          avpro_glide = 0, avpro_glide_alp = 0, avpro_glide_tl = 0, avpro_glide_btl = 0)

    ## Transferring danger ratings
    if(ElevBand=="ALL" | ElevBand=="ALP") {TblPlot$dng_alp <- HzdAssessObj$HzdAssess$alp_hzd}
    if(ElevBand=="ALL" | ElevBand=="TL")  {TblPlot$dng_tl <- HzdAssessObj$HzdAssess$tl_hzd}
    if(ElevBand=="ALL" | ElevBand=="BTL") {TblPlot$dng_btl <- HzdAssessObj$HzdAssess$btl_hzd}

    ## Transfer avalanche problems
    if (nrow(HzdAssessObj$AvProb)>0) {

      if(ElevBand!="ALL") {warning("Elevation specific avalanche problems not yet implemented for Infoex avalanche hazard objectives.")}

      TblAvProb <- HzdAssessObj$AvProb

      for (i in 1:nrow(TblAvProb)) {

        HzdAssessUUID <- TblAvProb$infoexhzdassess_uuid[i]

        if (TblAvProb$character[i]=="LOOSE_DRY_AVALANCHE") {
          TblPlot$avpro_ldry[TblPlot$id==HzdAssessUUID] <- 1
        } else if (TblAvProb$character[i]=="LOOSE_WET_AVALANCHE") {
          TblPlot$avpro_lwet[TblPlot$id==HzdAssessUUID] <- 1
        } else if (TblAvProb$character[i]=="WET_SLAB"| TblAvProb$character[i]=="Wet Slab") {
          TblPlot$avpro_wets[TblPlot$id==HzdAssessUUID] <- 1
        } else if (TblAvProb$character[i]=="CORNICE") {
          TblPlot$avpro_corn[TblPlot$id==HzdAssessUUID] <- 1
        } else if (TblAvProb$character[i]=="WIND_SLAB" | TblAvProb$character[i]=="Wind Slab") {
          TblPlot$avpro_wind[TblPlot$id==HzdAssessUUID] <- 1
        } else if (TblAvProb$character[i]=="STORM_SLAB" | TblAvProb$character[i]=="Storm Slab") {
          TblPlot$avpro_storm[TblPlot$id==HzdAssessUUID] <- 1
        } else if (TblAvProb$character[i]=="PERSISTENT_SLAB" | TblAvProb$character[i]=="Persistent Slab") {
          TblPlot$avpro_pers[TblPlot$id==HzdAssessUUID] <- 1
        } else if (TblAvProb$character[i]=="DEEP_PERSISTENT_SLAB" | TblAvProb$character[i]=="Persistent Deep Slab") {
          TblPlot$avpro_dpers[TblPlot$id==HzdAssessUUID] <- 1
        } else if (TblAvProb$character[i]=="GLIDE" | TblAvProb$character[i]=="Glide") {
          TblPlot$avpro_glide[TblPlot$id==HzdAssessUUID] <- 1
        } else {
          stop(paste0("Unknown avalanche character type: ", TblAvProb$character[i]))
        }

      }#end for

      rm(i, TblAvProb, HzdAssessUUID)

    }


  } else if (names(HzdAssessObj)[1]=="Bulletins") {

    ## Creating of TblPlot
    TblPlot <- data.frame(id = HzdAssessObj$Bulletins$BULLETIN_ID,
                          date = HzdAssessObj$Bulletins$PUBLISH_DATE,
                          dng_alp = NA, dng_tl = NA, dng_btl = NA,
                          avpro_ldry = 0,  avpro_ldry_alp = 0,  avpro_ldry_tl = 0,  avpro_ldry_btl = 0,
                          avpro_lwet = 0,  avpro_lwet_alp = 0,  avpro_lwet_tl = 0,  avpro_lwet_btl = 0,
                          avpro_wets = 0,  avpro_wets_alp = 0,  avpro_wets_tl = 0,  avpro_wets_btl = 0,
                          avpro_corn = 0,  avpro_corn_alp = 0,  avpro_corn_tl = 0,  avpro_corn_btl = 0,
                          avpro_wind = 0,  avpro_wind_alp = 0,  avpro_wind_tl = 0,  avpro_wind_btl = 0,
                          avpro_storm = 0, avpro_storm_alp = 0, avpro_storm_tl = 0, avpro_storm_btl = 0,
                          avpro_pers = 0,  avpro_pers_alp = 0,  avpro_pers_tl = 0,  avpro_pers_btl = 0,
                          avpro_dpers = 0, avpro_dpers_alp = 0, avpro_dpers_tl = 0, avpro_dpers_btl = 0,
                          avpro_glide = 0, avpro_glide_alp = 0, avpro_glide_tl = 0, avpro_glide_btl = 0)

    ## Transferring danger ratings
    DngRating <- HzdAssessObj$DngRating
    for (i in 1:nrow(TblPlot)) {
      BullID <- TblPlot$id[i]
      if(ElevBand=="ALL" | ElevBand=="ALP") {TblPlot$dng_alp[i] <- as.character(DngRating$Alp$DAY0[DngRating$Alp$BULLETIN_ID==BullID])}
      if(ElevBand=="ALL" | ElevBand=="TL") {TblPlot$dng_tl[i] <- as.character(DngRating$Tl$DAY0[DngRating$Tl$BULLETIN_ID==BullID])}
      if(ElevBand=="ALL" | ElevBand=="BTL") {TblPlot$dng_btl[i] <- as.character(DngRating$Btl$DAY0[DngRating$Btl$BULLETIN_ID==BullID])}
    }
    rm(DngRating, i)

    ## Transferring avalanche problems - ALP
    if(ElevBand=="ALL" | ElevBand=="ALP") {
      if (nrow(HzdAssessObj$AvProblems$Alp)>0) {
        TblAvProb <- HzdAssessObj$AvProblems$Alp
        for (i in 1:nrow(TblAvProb)) {
          HzdAssessUUID <- TblAvProb$BULLETIN_ID[i]
          if (TblAvProb$CHARACTER[i]=="Loose Dry") {
            TblPlot$avpro_ldry[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_ldry_alp[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Loose Wet") {
            TblPlot$avpro_lwet[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_lwet_alp[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Wet Slabs") {
            TblPlot$avpro_wets[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_wets_alp[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Cornices") {
            TblPlot$avpro_corn[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_corn_alp[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Wind Slabs") {
            TblPlot$avpro_wind[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_wind_alp[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Storm Slabs") {
            TblPlot$avpro_storm[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_storm_alp[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Persistent Slabs") {
            TblPlot$avpro_pers[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_pers_alp[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Deep Persistent Slabs") {
            TblPlot$avpro_dpers[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_dpers_alp[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Glide") {
            TblPlot$avpro_glide[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_glide_alp[TblPlot$id==HzdAssessUUID] <- 1
          } else {
            stop(paste0("Unknown avalanche problem type: ", TblAvProb$CHARACTER[i]))
          }
        }
        rm(i, TblAvProb, HzdAssessUUID)
      } # end if for alp
    }

    ## Transferring avalanche problems - TL
    if(ElevBand=="ALL" | ElevBand=="TL") {
      if (nrow(HzdAssessObj$AvProblems$Tl)>0) {
        TblAvProb <- HzdAssessObj$AvProblems$Tl
        for (i in 1:nrow(TblAvProb)) {
          HzdAssessUUID <- TblAvProb$BULLETIN_ID[i]
          if (TblAvProb$CHARACTER[i]=="Loose Dry") {
            TblPlot$avpro_ldry[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_ldry_tl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Loose Wet") {
            TblPlot$avpro_lwet[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_lwet_tl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Wet Slabs") {
            TblPlot$avpro_wets[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_wets_tl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Cornices") {
            TblPlot$avpro_corn[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_corn_tl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Wind Slabs") {
            TblPlot$avpro_wind[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_wind_tl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Storm Slabs") {
            TblPlot$avpro_storm[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_storm_tl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Persistent Slabs") {
            TblPlot$avpro_pers[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_pers_tl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Deep Persistent Slabs") {
            TblPlot$avpro_dpers[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_dpers_tl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Glide") {
            TblPlot$avpro_glide[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_glide_tl[TblPlot$id==HzdAssessUUID] <- 1
          } else {
            stop(paste0("Unknown avalanche problem type: ", TblAvProb$CHARACTER[i]))
          }
        }
        rm(i, TblAvProb, HzdAssessUUID)
      } # end if for tl
    }

    ## Transferring avalanche problems - BTL
    if(ElevBand=="ALL" | ElevBand=="BTL") {
      if (nrow(HzdAssessObj$AvProblems$Btl)>0) {
        TblAvProb <- HzdAssessObj$AvProblems$Btl
        for (i in 1:nrow(TblAvProb)) {
          HzdAssessUUID <- TblAvProb$BULLETIN_ID[i]
          if (TblAvProb$CHARACTER[i]=="Loose Dry") {
            TblPlot$avpro_ldry[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_ldry_btl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Loose Wet") {
            TblPlot$avpro_lwet[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_lwet_btl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Wet Slabs") {
            TblPlot$avpro_wets[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_wets_btl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Cornices") {
            TblPlot$avpro_corn[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_corn_btl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Wind Slabs") {
            TblPlot$avpro_wind[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_wind_btl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Storm Slabs") {
            TblPlot$avpro_storm[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_storm_btl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Persistent Slabs") {
            TblPlot$avpro_pers[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_pers_btl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Deep Persistent Slabs") {
            TblPlot$avpro_dpers[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_dpers_btl[TblPlot$id==HzdAssessUUID] <- 1
          } else if (TblAvProb$CHARACTER[i]=="Glide") {
            TblPlot$avpro_glide[TblPlot$id==HzdAssessUUID] <- 1
            TblPlot$avpro_glide_btl[TblPlot$id==HzdAssessUUID] <- 1
          } else {
            stop(paste0("Unknown avalanche problem type: ", TblAvProb$CHARACTER[i]))
          }
        }
        rm(i, TblAvProb, HzdAssessUUID)
      } # end if for btl
    }

  }

  ## assign date range for chart
  if(is.na(DateStart)) {DateStart <- min(TblPlot$date)}
  if(is.na(DateEnd))   {DateEnd   <- max(TblPlot$date)}

  DateStart <- as.Date(DateStart)
  DateEnd   <- as.Date(DateEnd)

  ## SET UP PLOT
  ylim_min <- 0
  ylim_max <- 3
  HzdRatBase <- 0
  HzdRatHeight <- 1

  if (plotAvProb){

    if (NoGlideNoCornice) {ylim_min <- 2.75} else {ylim_min <- 0.75}
    if (!is.null(SHObsDates)){ylim_max <- 12.5} else {ylim_max <- 11.5}
    HzdRatBase <- 10
    HzdRatHeight <- 0.5

  }
  ylim <- c(ylim_min, ylim_max)

  plot(NULL, xlim=c(DateStart, DateEnd), ylim=ylim, yaxt="n", xaxt="n", ylab="", xlab="")
  addTimeGridAndHighlight(TblPlot$date, WithXAxis=WithXAxis,
                          HighlightDateStart=HighlightDateStart, HighlightDateEnd=HighlightDateEnd,
                          HighlightCol=HighlightCol, cex.axis=cex.axis, cex.lab=cex.lab)
  if (plotAvProb){
    abline(h=c(1:9), lty=3, col="dark grey")
    axis(2, at=c(1:9), labels=c("Glide", "Corn", "WetS", "LWet", "DPers", "Pers", "Storm", "Wind", "LDry"), cex.axis=cex.axis*0.5, cex.lab=cex.lab, las=1)
  }

  title(main = main)

  box()

  ## add Av problems
  if (plotAvProb){
    points(TblPlot$date, TblPlot$avpro_ldry*9,  pch=19, col=getAvCharColor("LOOSE_DRY_AVALANCHE"), cex=cex.points)
    points(TblPlot$date, TblPlot$avpro_wind*8,  pch=19, col=getAvCharColor("WIND_SLAB"), cex=cex.points)
    points(TblPlot$date, TblPlot$avpro_storm*7, pch=19, col=getAvCharColor("STORM_SLAB"), cex=cex.points)
    points(TblPlot$date, TblPlot$avpro_pers*6,  pch=19, col=getAvCharColor("PERSISTENT_SLAB"), cex=cex.points)
    points(TblPlot$date, TblPlot$avpro_dpers*5, pch=19, col=getAvCharColor("DEEP_PERSISTENT_SLAB"), cex=cex.points)
    points(TblPlot$date, TblPlot$avpro_lwet*4,  pch=19, col=getAvCharColor("LOOSE_WET_AVALANCHE"), cex=cex.points)
    points(TblPlot$date, TblPlot$avpro_wets*3,  pch=19, col=getAvCharColor("WET_SLAB"), cex=cex.points)
    if(NoGlideNoCornice) {
      points(TblPlot$date, TblPlot$avpro_corn*2,  pch=19, col=getAvCharColor("CORNICE"), cex=cex.points)
      points(TblPlot$date, TblPlot$avpro_glide*1, pch=19, col=getAvCharColor("GLIDE"), cex=cex.points)
    }
  }

  ## add hazard ratings
  for (DateIndex in 1:nrow(TblPlot)) {
    rect(TblPlot$date[DateIndex]-0.5, HzdRatBase+2*HzdRatHeight, TblPlot$date[DateIndex]+0.5, HzdRatBase+3*HzdRatHeight, col=getHzdColor(TblPlot$dng_alp[DateIndex]))
    rect(TblPlot$date[DateIndex]-0.5, HzdRatBase+1*HzdRatHeight, TblPlot$date[DateIndex]+0.5, HzdRatBase+2*HzdRatHeight, col=getHzdColor(TblPlot$dng_tl[DateIndex]))
    rect(TblPlot$date[DateIndex]-0.5, HzdRatBase,                TblPlot$date[DateIndex]+0.5, HzdRatBase+HzdRatHeight,   col=getHzdColor(TblPlot$dng_btl[DateIndex]))
  }
  if(plotElevLabels){
    if (plotAvProb){
      axis(2, at=c(10.25,10.75,11.25), labels=c("BTL", "TL", "ALP"), cex=cex.axis*0.25, las=1)
    } else {
      axis(2, at=c(0.5, 1.5, 2.5), labels=c("BTL", "TL", "ALP"), cex=cex.axis*0.25, las=1)
    }
  }

  ## add Av observations
  if(!is.null(AvObs)){

    warning("Adding avalanche observations to the avalanche problem plot needs to be updated.")

    # AvCharacters <- c("LOOSE_DRY_AVALANCHE",
    #                   "LOOSE_WET_AVALANCHE",
    #                   "WET_SLAB",
    #                   "CORNICE",
    #                   "WIND_SLAB",
    #                   "STORM_SLAB",
    #                   "PERSISTENT_SLAB",
    #                   "DEEP_PERSISTENT_SLAB",
    #                   "GLIDE")
    #
    # for (i in 1:length(AvCharacters)){
    #   plotDF          <- AvObs %>% filter(character == AvCharacters[i])
    #   plotDF$plotChar <- 1
    #   points(plotDF$date, plotDF$plotChar*9-i, pch=0, cex=cex.points)
    # }
  }

  ## Add observed surface hoar
  if(!is.null(SHObsDates)){
    text(labels="v", x=SHObsDates, y=8.5, adj=0.5, col="darkred")
  }

}#end function
