#' Plot snow height and precipitation chart
#'
#' Plot snow height and precipitation chart.
#' @param TblWx Dataframe with Wx observations. The following columns are needed: date, hn24, rainfall, hs, snow_form_1 (optional), snow_form_2 (optional)
#' @param WithXAxis Switch for plotting x-axis (default=TRUE).
#' @param DateStart Start date for plot. If not provided, the function takes the date range of TblWx (default=NA).
#' @param DateEnd End date for plot. If not provided, the function takes the date range of TblWx (default=NA).
#' @param HighlightDateStart Start of highlighted time period (default=NA).
#' @param HighlightDateEnd End of highlighted time period (default=HighlightDateStart).
#' @param HighlightCol Color for highlighted time period (default="light grey").
#' @param HsOffset Offset for HS plot (default=100).
#' @param Hn24Offset Offset for HN24 plot (default=110).
#' @param Hn24Scale Scale for HN24 plot(default=2).
#' @param SHShow Switch for showing SH icons on top of snowpack in chart (Default=TRUE).
#' @param SHOffset Offset for SH plot(default=15).
#' @param HsCol Color for HS (default="black").
#' @param Hn24Col Color for HN24 (default="grey").
#' @param RainCol Color for Rain (default="grey10").
#' @param cex.axis cex.axis (default=1.0).
#' @param cex.lab cex.lab (default=1.0).
#' @param WxSiteName WxSiteName (default=NULL).
#'
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' require(SarpGeneralVis)
#'
#' Operation <- "MPS"
#' WxSite <- "Lodge Weather Plot"
#'
#' DateStart <- "2015-12-15"
#' DateEnd <- "2016-04-15"
#'
#' ## Get InfoExWxObs for a specific Wx station and time period
#' TblWx <- getInfoExWxObs(Operation, WxSiteName=WxSite, DateStart=DateStart, DateEnd=DateEnd)
#'
#' ## Standard plot
#' plotTSWxSnow(TblWx)
#'
#' ## Adjust colors of HS, HN24, Rain
#' plotTSWxSnow(TblWx, HsCol="darkred", Hn24Col="cyan2", RainCol="darkblue")
#'
#' @export

plotTSWxSnow <- function(TblWx,
                       WithXAxis=T,
                       DateStart=NA,
                       DateEnd=NA,
                       HighlightDateStart=NA,
                       HighlightDateEnd=HighlightDateStart,
                       HighlightCol="light grey",
                       HsOffset=100,
                       Hn24Offset=110,
                       Hn24Scale=2,
                       SHShow=TRUE,
                       SHOffset=15,
                       HsCol="black",
                       Hn24Col="grey",
                       RainCol="grey10",
                       cex.axis=1.0,
                       cex.lab=1.0,
                       WxSiteName=NULL) {

  ## Check
  if(SHShow) {
    if (!(("snow_form_1" %in% names(TblWx))) | !(("snow_form_2" %in% names(TblWx)))) {
      warning("Columns 'snow_form_1' and 'snow_form_2' are required to show the surface hoar surface forms", immediate. = T)
      SHShow <- FALSE
    }
  }

  print("... starting plotTSWxSnow")

  # assign date range for chart
  if(is.na(DateStart)) {DateStart <- min(TblWx$date)}
  if(is.na(DateEnd))   {DateEnd   <- max(TblWx$date)}

  DateStart <- as.Date(DateStart)
  DateEnd <- as.Date(DateEnd)

  ## Function for adding surface form symbols
  addSfcForm <- function(TblWx, SfcForm, Label, SfcForm1Col="red", SfcForm2Col="red", Offset=15, cex=0.75, cex.axis=cex.axis, cex.lab=cex.lab) {

    SfcForm1 <- TblWx[grep(SfcForm, TblWx$snow_form_1), c("date", "snow_form_1")]
    SfcForm1 <- SfcForm1[!duplicated(SfcForm1), ]
    SfcForm2 <- TblWx[grep(SfcForm, TblWx$snow_form_2), c("date", "snow_form_2")]
    SfcForm2 <- SfcForm2[!duplicated(SfcForm2), ]

    DailyHS <- aggregate(hs~date, data=TblWx, max)

    if(nrow(SfcForm2)>0) {
      for(Index in 1:nrow(SfcForm2)){
        text(DailyHS$date[DailyHS$date==SfcForm2$date[Index]], DailyHS$hs[DailyHS$date==SfcForm2$date[Index]]+Offset, Label, adj=0.5, cex=cex, font=2, col=SfcForm2Col)
      }
    }
    if(nrow(SfcForm1)>0) {
      for(Index in 1:nrow(SfcForm1)){
        text(DailyHS$date[DailyHS$date==SfcForm1$date[Index]], DailyHS$hs[DailyHS$date==SfcForm1$date[Index]]+Offset, Label, adj=0.5, cex=cex, font=2, col=SfcForm1Col)
      }
    }
  }#end function

  ## Function for adding HN24 and rainfall bar chart
  addHn24AndRainBars <- function(DF, Offset=0, Scale=1, WithAxis=T, AxisInterval=10, Hn24Col=Hn24Col, RainCol=RainCol, cex.axis=cex.axis, cex.lab=cex.lab) {

    ## Replace NA with zeros
    DF$hn24[is.na(DF$hn24)] <- 0
    DF$rainfall[is.na(DF$rainfall)] <- 0

    ## Plotting squares
    for (DateIndex in 1:nrow(DF)) {
      rect(DF$date[DateIndex]-0.5, 0+Offset, DF$date[DateIndex]+0.5, DF$hn24[DateIndex]*Scale+Offset, col=Hn24Col, lwd=1)
      rect(DF$date[DateIndex]-0.5, 0+Offset+DF$hn24[DateIndex]*Scale, DF$date[DateIndex]+0.5, DF$rainfall[DateIndex]*Scale+Offset+DF$hn24[DateIndex]*Scale, col=RainCol, lwd=1)
    }

    ## Add axis
    if(WithAxis) {
      AxisMin <- 0
      AxisMax <- ((max(DF$hn24) %% AxisInterval)+1) * AxisInterval
      TickLoc <- seq(Offset, Offset+AxisMax*Scale, by=AxisInterval*Scale)
      TickLabels <- seq(AxisMin, AxisMax, by=AxisInterval)
      axis(4, at=TickLoc, labels=TickLabels, las=1, cex.axis=cex.axis)
      mtext("NH24 / Rainfall", 4, 2, cex=cex.lab*0.75)
    }

  }#end function

  ## Data manipulation
  DailySnow <- aggregate(hs~date, data=TblWx, max)
  DailySnow <- merge(DailySnow, aggregate(hn24~date, data=TblWx, sum), all.x=T)
  if(sum(is.na(TblWx$rainfall)) == nrow(TblWx)){
     DailySnow$rainfall <- NA
  }else{
     DailySnow <- merge(DailySnow, aggregate(rainfall~date, data=TblWx, sum), all.x=T)
  }

  ## set up plot
  plot(DailySnow$date, DailySnow$hs, t="n", xlim=c(as.Date(DateStart), as.Date(DateEnd)), ylim=c(HsOffset, max(DailySnow$hs)+10), xaxt="n", ylab="HS", xlab="", las=1, cex.axis=cex.axis, cex.lab=cex.lab)
  addTimeGridAndHighlight(DailySnow$date, WithXAxis=WithXAxis,
                          HighlightDateStart=HighlightDateStart, HighlightDateEnd=HighlightDateEnd,
                          HighlightCol=HighlightCol, cex.axis=cex.axis, cex.lab=cex.lab)
  abline(h=seq(0, 1000, by=50), lty=3, col="dark grey")
  box()
  if(!is.null(WxSiteName)){
    mtext(text = WxSiteName, side = 3, line = 0.2, adj = 1, cex = cex.axis*0.5)
  }

  ## plot data
  points(DailySnow$date, DailySnow$hs, ty="l", col=HsCol)
  addHn24AndRainBars(DailySnow, Offset=Hn24Offset, Scale=Hn24Scale, Hn24Col=Hn24Col, RainCol=RainCol, cex.axis=cex.axis, cex.lab=cex.lab)
  if(SHShow) {addSfcForm(TblWx, "SH", "v", Offset=SHOffset)}

}
