#' Plot temperature
#'
#' Plot temperature.
#' @param TblWx Dataframe with Wx observations.
#' @param WithXAxis Switch for plotting x-axis (default=TRUE).
#' @param DateStart Start date for plot. If not provided, the function takes the date range of TblWx (default=NA).
#' @param DateEnd End date for plot. If not provided, the function takes the date range of TblWx (default=NA).
#' @param HighlightDateStart Start of highlighted time period (default=NA).
#' @param HighlightDateEnd End of highlighted time period (default=HighlightDateStart).
#' @param HighlightCol Color for highlighted time period (default="light grey").
#' @param plotType Type of temperature plot. Possible types are "Lines" and "Polygon" (default="Lines").
#' @param main Title of chart
#' @param cex.axis cex.axis (default=1.0).
#' @param cex.lab cex.lab (default=1.0).
#' @param Verbose Swtich for turning messages on or off (default value FALSE).
#'
#' @examples
#' require(SarpGPSTools)
#' require(SarpGPSToolsPrivate)
#' require(SarpGeneralVis)
#'
#' Operation <- "NEH"
#' DateStart <- "2015-01-01"
#' DateEnd <- "2015-03-31"
#'
#' ## Get InfoExWxObs for a specific Wx station and time period
#' TblWx <- getInfoExWxObs(Operation=Operation, WxSiteName="Shames", DateStart=DateStart, DateEnd=DateEnd)
#'
#' ## Standard plot with lines for min and max temperature
#' plotTSWxTemp(TblWx, plotType = "Lines")
#'
#' ## Adjusted size of axis text and axis labels
#' plotTSWxTemp(TblWx, plotType = "Lines", cex.axis=1.5, cex.lab=1.5)
#'
#' ## Standard plot with shaded polygon for min and max temperature and mean temperature as line
#' plotTSWxTemp(TblWx, plotType = "Polygon")
#'
#' ## Color one part of a time series and shade the rest, adjust the color of the highlight
#' plotTSWxTemp(TblWx, plotType = "Polygon", HighlightDateStart=as.Date("2015-02-18"), HighlightCol = "#C0C0C050")
#'
#' @export

plotTSWxTemp <- function(TblWx,
                       WithXAxis=T,
                       DateStart=NA,
                       DateEnd=NA,
                       HighlightDateStart=NA,
                       HighlightDateEnd=HighlightDateStart,
                       HighlightCol="light grey",
                       plotType="Lines",
                       main="",
                       cex.axis=1.0,
                       cex.lab=1.0,
                       Verbose=F) {


  print("... starting plotTSWxTemp")

  ## assign date range for chart
  if(is.na(DateStart)) {DateStart <- min(TblWx$date)}
  if(is.na(DateEnd))   {DateEnd   <- max(TblWx$date)}

  ## data manipulation
  DailyTemp <- aggregate(temp_min~date, data=TblWx, min)
  DailyTemp <- merge(DailyTemp, aggregate(temp_max~date, data=TblWx, max))
  DailyTemp <- merge(DailyTemp, aggregate(temp_pres~date, data=TblWx, mean))
  DailyTemp$temp_mean <- rowMeans(DailyTemp[,2:3], na.rm=TRUE)

  ## set up plot
  plot(DailyTemp$date, DailyTemp$temp_min, col="blue", t="n", xlim=c(DateStart, DateEnd), ylim=c(min(DailyTemp$temp_min)-2, max(DailyTemp$temp_max)+2), xaxt="n", xlab="", ylab=expression(Temperature~(degree~C)), las=1, cex.axis=cex.axis, cex.lab=cex.lab)
  addTimeGridAndHighlight(DailyTemp$date, WithXAxis=WithXAxis, HighlightDateStart=HighlightDateStart, HighlightDateEnd=HighlightDateEnd, HighlightCol=HighlightCol, cex.axis=cex.axis, cex.lab=cex.lab)
  abline(h=seq(-50, 50, by=5), lty=3, col="dark grey")
  box()
  title(main = main)

  ## plot data
  if(plotType=="Lines"){
    lines(DailyTemp$date, DailyTemp$temp_min, col="blue")
    lines(DailyTemp$date, DailyTemp$temp_max, col="red")
    lines(DailyTemp$date, DailyTemp$temp_pres, col="black")
  }

  #DailyTemp <- na.omit(DailyTemp)

  if(plotType=="Polygon"){

    InsideDateRange <- T

    ## Adjust end date
    if((is.na(HighlightDateStart))|(HighlightDateStart > max(TblWx$date))){
      HighlightDateStart=max(TblWx$date)
    }

    pos <- match(as.Date(HighlightDateStart), DailyTemp$date)

    if (Verbose) {
      print("1:")
      print(as.Date(HighlightDateStart))
      print(pos)
    }

    if(is.na(pos)){
      ## Covers gaps within temp tie series
      if(HighlightDateStart > min(TblWx$date)){
        pos <- match(as.Date(HighlightDateStart-1), DailyTemp$date)
        if(Verbose) {
          print("2:")
          print(as.Date(HighlightDateStart-1))
          print(pos)
        }
        ## Cover highlight date before data range
      } else {
        pos <- 1
        InsideDateRange <- F
      }
    }

    ## Plotting background
    polx2 <- c(DailyTemp$date[pos:nrow(DailyTemp)],rev(DailyTemp$date[pos:nrow(DailyTemp)]))
    poly2 <- c(DailyTemp$temp_min[pos:nrow(DailyTemp)],rev(DailyTemp$temp_max[pos:nrow(DailyTemp)]))

    polygon(polx2,poly2,col="#E8E8E850", border=NA)
    lines(DailyTemp$date, DailyTemp$temp_min, col="grey")
    lines(DailyTemp$date, DailyTemp$temp_max, col="grey")
    lines(DailyTemp$date, DailyTemp$temp_pres, col="grey")

    ## Plotting active range
    if (InsideDateRange) {
      polx1 <- c(DailyTemp$date[1:pos],rev(DailyTemp$date[1:pos]))
      poly1 <- c(DailyTemp$temp_min[1:pos],rev(DailyTemp$temp_max[1:pos]))

      polygon(polx1,poly1,col="#CBCBCB50", border=NA)
      lines(DailyTemp$date[1:pos], DailyTemp$temp_min[1:pos], col="blue")
      lines(DailyTemp$date[1:pos], DailyTemp$temp_max[1:pos], col="red")
      lines(DailyTemp$date[1:pos], DailyTemp$temp_pres[1:pos], col="black")
    }

  }

  ## Zero degree line
  abline(h=0, lty=3, lwd=1)
}
