#' Plot average Wind Direction and Speed
#'
#' @param TblWx Dataframe with Wx observations.
#' @param WithXAxis Switch for plotting x-axis (default=TRUE).
#' @param DateStart Start date for plot. If not provided, the function takes the date range of TblWx (default=NA).
#' @param DateEnd End date for plot. If not provided, the function takes the date range of TblWx (default=NA).
#' @param ShowTestArrows Adds X & Y arrows to allow for visual adjustment of ScaleX and ScaleY (default=F)
#' @param ScaleX Scale used to adjust the wind direction arrows x-axis (default is 2.5)
#' @param ScaleY Scale used to adjust the wind direction arrows y-axis (default is 0.1)
#' @param HighlightDateStart Start of highlighted time period (default=NA).
#' @param HighlightDateEnd End of highlighted time period (default=HighlightDateStart).
#' @param HighlightCol Color for highlighted time period (default="light grey").
#' @param plotType Type of plot. Possible types are "Lines" and "Polygon" (default="Lines").
#' @param main Title of chart
#' @param cex.axis cex.axis (default=1.0).
#' @param cex.lab cex.lab (default=1.0).
#' @export


plotTSWxWind <- function(TblWx,
                         WithXAxis=T,
                         DateStart=NA,
                         DateEnd=NA,
                         ShowTestArrows=F,
                         ScaleX=2.5,
                         ScaleY=0.1,
                         Level=0.2,
                         HighlightDateStart=NA,
                         HighlightDateEnd=HighlightDateStart,
                         HighlightCol="light grey",
                         plotType="Lines",
                         main="",
                         cex.axis=1.0,
                         cex.lab=1.0,
                         Verbose=F) {


  print("... starting plotTSWxWind")

  # TblWx <- TblWxWind
  # WithXAxis=T
  # DateStart=as.Date("2012-10-01")
  # DateEnd=as.Date("2013-10-01")
  # ShowTestArrows=T
  # ScaleX=2.5
  # ScaleY=0.1
  # Level=0.2
  # HighlightDateStart=NA
  # HighlightDateEnd=HighlightDateStart
  # HighlightCol="light grey"
  # plotType="Lines"
  # main=""
  # cex.axis=1.0
  # cex.lab=1.0

  ## assign date range for chart if no range specified
  if(is.na(DateStart)) {DateStart <- min(TblWx$date)}
  if(is.na(DateEnd))   {DateEnd   <- max(TblWx$date)}



  # converts wind direction in degrees to radians
  TblWx$dir_rad <- TblWx$dir/180*pi

  # Plots Wind Speed for date range
  plot(TblWx$date, TblWx$spd, t="n", xlim=c(as.Date(DateStart), as.Date(DateEnd)), ylim=c(0, max(TblWx$spd, na.rm = T)), xaxt="n", ylab="Wind Spd (m/s)", xlab="", las=1, cex.axis=cex.axis, cex.lab=cex.lab)
  addTimeGridAndHighlight(TblWx$date, WithXAxis=WithXAxis,
                          HighlightDateStart=HighlightDateStart, HighlightDateEnd=HighlightDateEnd,
                          HighlightCol=HighlightCol, cex.axis=cex.axis, cex.lab=cex.lab)
  abline(h=seq(0, 1000, by=50), lty=3, col="dark grey")
  box()
  lines(TblWx$date, TblWx$spd)

  # Plots wind direction on same plot as wind speed
  arrows(x0=TblWx$date, y0=TblWx$spd, x1=TblWx$date+ScaleX*sin(TblWx$dir_rad), y1=TblWx$spd+ScaleY*cos(TblWx$dir_rad), code = 2, length=0.05, angle = 135,  lwd=2, col = ifelse(TblWx$spd >= 4, "red", "black"))
  ## Test - Plots a sample x & y arrow to help adjust ScaleX and ScaleY so that each arrow is the same length (visually)
  if (ShowTestArrows) {
    class(TblWx) <- "data.frame"
    temp <- data.frame (x = TblWx$date[c(1,1)],
                        y = rep(Level+0.5, 2),
                        dir_rad = c(0, pi/2))


    arrows(x0=temp$x, y0=temp$y, x1=temp$x+ScaleX*sin(temp$dir_rad), y1=temp$y+ScaleY*cos(temp$dir_rad), code = 2, length=0.05, angle = 135,  lwd=2)

  }
} # End Function
