#' Plots terrain in either grey scale or with terrain colors based on DEM
#'
#' Plots terrain in either grey scale or with terrain colors based on DEM
#' @param DEM raster with elevation data.
#' @param Type Type of terrain plot. Default is 'Elev'. Other option is 'Slope'.
#' @param Elev.Clr.Type Type of terrain shading. Default is 'Terrain'. Other option is 'Monochrome'
#' @param Elev.Hsh Switch for showing hillshade in the background. Default is TRUE
#' @param Elev.Min Start of elevation range. Default value is 0. Only used when applying monochrome scale.
#' @param Elev.Max End of elevation range. Default value is 4000.
#' @param Elev.Step Elevation intervals for terrain shading. Default is 100m.
#' @param Elev.BtlTl Elevation of BTL-TL boundary. Default is 1800m
#' @param Elev.TlAlp Elevation of TL-ALP boundary. Default is 2400m.
#' @param Elev.Clr Array of two color for monochromatic elevation shading. Default are two grey shades c("#E8E8E8", "#808080").
#' @param Elev.Cont Switch for adding elevation contour lines. Default is FALSE.
#' @param Elev.Cont.Clr Color of elevation contours.
#' @param Elev.WithHsh.alpha Transparency of elevation shading layer if hillshade is included. Default is 0.35,
#' @param Slp.Hsh Switch for showing hillshade in the background of slope shading. Default is TRUE,
#' @param Slp.Int Breaks for slope shading. Default is c(0, 20, 25, 30, 35, 40, 45, 50, 90)
#' @param Slp.Clr Array of two color for monochromatic elevation shading. Default is NA.
#' @param Slp.ClrBrewer Name of Colorbrewer scale for slope shading. Default is "OrRd"
#' @param Slp.Cont Switch for adding slope incline contour lines. Default is FALSE.
#' @param Slp.Cont.Clr Color of elevation contours.
#' @param Slp.WithHsh.alpha Transparency of slope shading layer if hillshade is included. Default is 0.35,
#' @param alpha Transparency of entire chart. Default is 1.
#' @param add Switch for whether plot should be added to existing plot (default=F)
#' @param ... Any other parameters that can be added to the plot function.
#'
#' @examples
#' require(SarpGeneralVis)
#'
#' ## Get test DEM
#' DEM <- getDEM_BCAB()
#'
#' ## Default: hilshade with terrain colors
#' plotTerrain(DEM)
#'
#' ## Hillshade only
#' plotTerrain(DEM, Elev.WithHsh.alpha=0)
#'
#' ## Grey scale with default 200m intervals
#' plotTerrain(DEM, Elev.Clr.Type="Monochrome", Elev.Step=500, Elev.Hsh=F)
#'
#' @export

plotTerrain <- function (DEM, Type = "Elev",
                          Elev.ClrType = "Terrain", Elev.Hsh = T, Elev.Min = 0, Elev.Max = 4000, Elev.Step = 100, Elev.BtlTl = 1800, Elev.TlAlp = 2400, Elev.Clr = c("#E8E8E8", "#808080"), Elev.Cont = F, Elev.Cont.Clr = NA, Elev.WithHsh.alpha = 0.35,
                          Slp.Hsh = T, Slp.Int = c(0, 20, 25, 30, 35, 40, 45, 50, 90), Slp.Clr = NA, Slp.ClrBrewer = "OrRd", Slp.Cont = F, Slp.Cont.Clr = NA, Slp.WithHsh.alpha = 0.35,
                          alpha = 1, add = F, ...) {

  ## Function for transparent color scale
  colorRampAlpha <- function(..., n, alpha) {
    colors <- colorRampPalette(...)(n)
    paste(colors, sprintf("%x", ceiling(255*alpha)), sep="")
  }

  ## Function for terrain color scale
  colorTerrain <- function(BtlTl = 1800, TlAlp = 2200, Step = 200, alpha = 0.35) {

    if (((TlAlp-BtlTl) %% Step) != 0) {stop("When using the terrain colors scale, the interval value must be an even portion of the difference between the elevations for BtlTl and TlAlp!")}

    Num_Btl <- BtlTl/Step - 1
    Num_Tl <- (TlAlp-BtlTl)/Step
    Num_Alp <- (5000-TlAlp)/Step - 1

    Col_Btl <- colorRampAlpha(c("#238443", "#238443"), n = Num_Btl, alpha = alpha)
    Col_Tl  <- colorRampAlpha(c("#238443", "#dfc27d", "#D8D8D8", "#FFFFFF"), n = Num_Tl, alpha = alpha)
    Col_Alp <- colorRampAlpha(c("#FFFFFF", "#FFFFFF"), n = Num_Alp, alpha = alpha)

    Col_Elev <- c(Col_Btl, Col_Tl, Col_Alp)

    breaks <- seq(0, 5000, Step)

    return(list(ramp = Col_Elev,
                breaks = breaks))

  }

  ## Check and modify inputs
  Type <- substr(tolower(Type), 1, 4)
  Elev.ClrType <- substr(tolower(Elev.ClrType), 1, 4)

  ## Proccessing DEM
  if (Elev.Hsh | Slp.Hsh | Type=="slop") {
    Slp <- terrain(DEM, opt="slope")
  }

  if (Elev.Hsh | Slp.Hsh) {
    Asp <- terrain(DEM, opt="aspect")
    Hsh <- hillShade(Slp, Asp, 40, 270)
  }

  if (Type=="slop") {
    Slp <- Slp/pi*180
  }

  ## Elevation with
  if (Type=="elev" & Elev.ClrType=="terr") {

    if (Elev.Hsh) {
      ColTerrain <- colorTerrain(BtlTl = Elev.BtlTl, TlAlp = Elev.TlAlp, Step = Elev.Step, alpha = Elev.WithHsh.alpha)
      plot(Hsh, col=grey(0:100/100, alpha = alpha), legend=F, add=add, ...)
      plot(DEM, breaks=ColTerrain$breaks, col=ColTerrain$ramp, legend=F, add=T)
    } else {
      ColTerrain <- colorTerrain(BtlTl = Elev.BtlTl, TlAlp = Elev.TlAlp, Step = Elev.Step, alpha = alpha)
      plot(DEM, breaks=ColTerrain$breaks, col=ColTerrain$ramp, legend=F, add=add, ...)
    }

    if(Elev.Cont) {
      if(is.na(Elev.Cont.Clr)) {Elev.Cont.Clr <- "#D3D3D3"}
      contour(DEM, col = Elev.Cont.Clr, add = T, lwd=0.5)
    }

  } else if (Type=="elev" & Elev.ClrType=="mono") {

    breaks <- seq(Elev.Min, Elev.Max, Elev.Step)
    n <- length(breaks)-1

    if (Elev.Hsh) {
      ColTerrain <- colorRampAlpha(Elev.Clr, n = n, alpha = Elev.WithHsh.alpha)
      plot(Hsh, col=grey(0:100/100, alpha = alpha), legend=F, add=add, ...)
      plot(DEM, breaks=breaks, col=ColTerrain, legend=F, add=T)

    } else {
      ColTerrain <- colorRampAlpha(Elev.Clr, n = n, alpha = alpha)
      plot(DEM, breaks=breaks, col=ColTerrain, legend=F, add=add, ...)

    }

    if(Elev.Cont) {
      if(is.na(Elev.Cont.Clr)) {Elev.Cont.Clr <- "#505050"}
      contour(DEM, col = Elev.Cont.Clr, add = T, lwd=0.5)
    }

  } else if (Type=="slop") {

    if (is.na(Slp.Clr)) {
      Slp.ClrScale <- RColorBrewer::brewer.pal(length(Slp.Int)-2, Slp.ClrBrewer)
      Slp.ClrScale <- c("#FFFFFF", Slp.ClrScale)
    } else {
      Slp.ClrScale <- colorRamp(Slp.Clr)(length(Slp.Int)-1)
    }

    if (Elev.Hsh) {
      Slp.ClrScale <- paste(Slp.ClrScale, sprintf("%x", ceiling(255*Slp.WithHsh.alpha)), sep="")
      plot(Hsh, col=grey(0:100/100, alpha = alpha), legend=F, add=add, ...)
      plot(Slp, breaks=Slp.Int, col=Slp.ClrScale, legend=F, add=T)
    } else {
      Slp.ClrScale <- paste(Slp.ClrScale, sprintf("%x", ceiling(255*alpha)), sep="")
      plot(Slp, breaks=Slp.Int, col=Slp.ClrScale, legend=F, add=add, ...)
    }

    if(Elev.Cont) {
      if(is.na(Elev.Cont.Clr)) {Elev.Cont.Clr <- "#505050"}
      contour(DEM, col = Elev.Cont.Clr, add = T, lwd=0.5)
    }

    if(Slp.Cont) {
      if(is.na(Slp.Cont.Clr)) {Slp.Cont.Clr <- "#505050"}
      contour(Slp, col = Slp.Cont.Clr, add = T, lwd=0.5)
    }


  }

}
