#' Plots topography of BC and AB as background
#'
#' Plots topography of BC and AB as background
#' @param ElevBy Size of elevation steps between 0 and 4000m (default: 500m)
#' @param ClrStart Color of lowest elevation category
#' @param ClrEnd Color of higest elevation category
#' @param ClrBoundary Color of provincial boundaries
#' @param WithBoundary Switch for including provincial boundaries (default=T)
#' @param add Switch for whether plot should be added to existing plot (default=F)
#'
#' @examples
#' require(SarpBulletinTools)
#' require(SarpBulletinToolsPrivate)
#'
#' BullRegionAvCan <- getBulletinRegionGeomSPDF(Season = 2018, Agency = "AvCan")
#' BullRegionPkCan <- getBulletinRegionGeomSPDF(Season = 2018, Agency = "PkCan")
#' BullRegionC <- getBulletinRegionGeomSPDF(Season = 2018, AsCentroid = T)
#'
#' ## Base plot
#' plot(BullRegionAvCan)
#' plotTerrain_BCAB(add=T)
#'
#' ## Bulletin regions
#' plot(BullRegionAvCan, col="#377eb880", lwd=2, add=T)
#' plot(BullRegionPkCan, col="#4daf4a80", lwd=2, add=T)
#'
#' ## Labels
#' text(BullRegionC, labels=BullRegionC$name)
#'
#' ## box
#' box()
#' title("Public avalanche bulletin regions in western Canada (2018)")
#'
#' @export

plotTerrain_BCAB <- function(Elev.Step=500, Elev.Clr=c("#E8E8E8", "#808080"), ClrBoundary="#606060", WithBoundaries=T, add=F, ...) {

  ## Load data
  DEM <- getDEM_BCAB()

  ## Plotting terrain
  plotTerrain(DEM = DEM,
              Type = 'Elev',
              Elev.ClrType = "Monochromatic",
              Elev.Hsh = F,
              Elev.Clr = Elev.Clr,
              Elev.Step = Elev.Step,
              add = add, ...)

  ## Add boundaries
  if(WithBoundaries) {
    load(url("http://data.avalancheresearch.ca/ProvBoundaries_BCAB.rda"))
    plot(ProvBoundaries_BCAB, border=ClrBoundary, add=T)
  }

}
