#' General base function for converting avalanche problems into spatial objects dataframe.
#'
#' General base function for converting avalanche problems into spatial objects dataframe.
#' @param AvProblems Avalanche problems dataframe with proper column names.
#' @param IDCol Name of datafram column with unique ID of avalanche problems.
#' @param Shape For specifying the shape of the blob. Values can be 'ellipse' (default) or 'square'.
#' @param NumPoints Number of point per quater section of ellipse. Default value is 50.
#' @return Avalanche Problems as spatial objects.

#' @export

convertAvProbToSPDFGeneral <- function(AvProblems, IDCol="uuid", Shape='ellipse', NumPoints=50) {

  ## Checks
  ## ******
  Shape <- toupper(Shape)

  RequiredColumns <- c(IDCol, "size_min", "size_typ", "size_max", "like_min", "like_typ", "like_max")
  RequiredIndex <- which(names(AvProblems) %in% RequiredColumns)
  if(length(RequiredIndex)<length(RequiredColumns)) {
    stop (paste0("AvProblems does not have the required columns: ", paste0(RequiredColumns, collapse = ", "), " The provided column names are: ", paste0(names(AvProblems), collapse = ", ")))
  }

  if (!isPackageInstalled("sp")) {stop("This function requires package 'sp' to be installed!")}
  if (!isPackageInstalled("rgeos")) {stop("This function requires package 'rgeos' to be installed!")}
  if (!isPackageInstalled("maptools")) {stop("This function requires package 'maptools' to be installed!")}

  ## Definition of functions
  ## ***********************
  EllipseY <- function(XAxis, YAxis, XCentre, YCentre, XValue) {
    EllipseY = YAxis * (1 - ((XValue - XCentre) / XAxis) ^ 2) ^ (1 / 2) + YCentre
  }

  EllipseX <- function(XAxis, XCentre, NumPoints, Point) {
    EllipseX = Point * XAxis / NumPoints + XCentre
  }

  ## Adding row names
  ## ****************
  row.names(AvProblems) <- AvProblems[,IDCol]

  ## Processing
  ## **********
  if (nrow(AvProblems)==0) {

    BlobSPDF <- NA
    warning("No avalanche problems in provided dataframe!", immediate. = T)

  } else {

    for (Index_AvProb in 1:nrow(AvProblems)) {

      ## Calculation of shape of blob
      ## ****************************

      XAxisL <- AvProblems$size_min[Index_AvProb]-AvProblems$size_typ[Index_AvProb]
      XAxisR <- AvProblems$size_max[Index_AvProb]-AvProblems$size_typ[Index_AvProb]
      YAxisU <- AvProblems$like_max[Index_AvProb]-AvProblems$like_typ[Index_AvProb]
      YAxisD <- AvProblems$like_min[Index_AvProb]-AvProblems$like_typ[Index_AvProb]

      if (Shape=="ELLIPSE") {

        TopL_X <- rev(EllipseX(XAxisL, AvProblems$size_typ[Index_AvProb], NumPoints, c(0:NumPoints)))
        if (XAxisL==0) {
          TopL_Y <- seq(AvProblems$like_typ[Index_AvProb], AvProblems$like_max[Index_AvProb], length.out=NumPoints+1)
        } else {
          TopL_Y <- EllipseY(XAxisL, YAxisU, AvProblems$size_typ[Index_AvProb], AvProblems$like_typ[Index_AvProb], TopL_X)
        }

        TopR_X <- EllipseX(XAxisR, AvProblems$size_typ[Index_AvProb], NumPoints, c(0:NumPoints))
        if (XAxisR==0) {
          TopR_Y <- seq(AvProblems$like_max[Index_AvProb], AvProblems$like_typ[Index_AvProb], length.out=NumPoints+1)
        } else {
          TopR_Y <- EllipseY(XAxisR, YAxisU, AvProblems$size_typ[Index_AvProb], AvProblems$like_typ[Index_AvProb], TopR_X)
        }

        BotR_X <- rev(EllipseX(XAxisR, AvProblems$size_typ[Index_AvProb], NumPoints, c(0:NumPoints)))
        if (XAxisR==0) {
          BotR_Y <- seq(AvProblems$like_typ[Index_AvProb], AvProblems$like_min[Index_AvProb], length.out=NumPoints+1)
        } else {
          BotR_Y <- EllipseY(XAxisR, YAxisD, AvProblems$size_typ[Index_AvProb], AvProblems$like_typ[Index_AvProb], BotR_X)
        }

        BotL_X <- EllipseX(XAxisL, AvProblems$size_typ[Index_AvProb], NumPoints, c(0:NumPoints))
        if (XAxisL==0) {
          BotL_Y <- seq(AvProblems$like_min[Index_AvProb], AvProblems$like_typ[Index_AvProb], length.out=NumPoints+1)
        } else {
          BotL_Y <- EllipseY(XAxisL, YAxisD, AvProblems$size_typ[Index_AvProb], AvProblems$like_typ[Index_AvProb], BotL_X)
        }

        Blob <- data.frame(x=c(TopL_X, TopR_X, BotR_X, BotL_X),
                           y=c(TopL_Y, TopR_Y, BotR_Y, BotL_Y))

      } else if (Shape=="SQUARE") {

        Blob <- data.frame(x=c(AvProblems$size_min[Index_AvProb], AvProblems$size_max[Index_AvProb], AvProblems$size_max[Index_AvProb], AvProblems$size_min[Index_AvProb], AvProblems$size_min[Index_AvProb]),
                           y=c(AvProblems$like_max[Index_AvProb], AvProblems$like_max[Index_AvProb], AvProblems$like_min[Index_AvProb], AvProblems$like_min[Index_AvProb], AvProblems$like_max[Index_AvProb]))

      } else {

        stop (paste0("Shape '", Shape, "' not supported. Use 'Ellipse' or 'Square' only!"))

      }

      ## Turning blob into spatial opjects
      ## *********************************
      BlobWKT <- paste0("POLYGON ((", paste(paste(Blob$x,  Blob$y), collapse = ", "), "))")
      BlobSpObj <- rgeos::readWKT(BlobWKT, id=AvProblems[Index_AvProb, IDCol])

      if (Index_AvProb==1) {
        BlobSpObjs  <- BlobSpObj
      } else {
        BlobSpObjs  <- maptools::spRbind(BlobSpObjs,  BlobSpObj)
      }

    }

    ## Combining spatial object with original data frame to spatial dataframe object
    BlobSPDF <- sp::SpatialPolygonsDataFrame(BlobSpObjs, AvProblems)

  }

  ## Output
  return (BlobSPDF)

}
