#' Derives avalanche season (typical operating season) info from winter year number.
#'
#' Derives avalanche season (typical operating season) info (start date, end date) from winter year number.
#' @param Season Single or array of winter season numbers.
#' @param Type Specifies whether a season is a full year ('AvYear'; default) or just the main winter months ('Operation') or a custom date range ('Custom'), which can be specified with the StartDate and EndDate parameters.
#' @param StartDate Start date for custom season length. Format needs to be MM-DD.
#' @param EndDate End date for custom season length. Format needs to be MM-DD.
#' @return List with single values or arrays of start and end dates for avalanche season
#'
#' @examples
#' Season <- c(2014, 2015)
#' SeasonInfo <- deriveAvSeasonInfo(Season)
#' StartDate <- SeasonInfo$StartDate
#' EndDate   <- SeasonInfo$EndDate
#'
#' @export

deriveAvSeasonInfo <- function(Season, Type="AvYear", StartDate=NA, EndDate=NA, OutputFormat=NA) {

  if(!is.na(OutputFormat)) {
    warning("Do not use 'OutputFormat' parameter anymore. Use 'Type' instead.", immediate. = TRUE)
    Type <- OutputFormat
  }

  if (tolower(Type)=="operation"){
    StartDateOut <- dates(paste(Season-1, "12", "01", sep="-"), format="y-m-d")
    EndDateOut   <- dates(paste(Season, "04", "15", sep="-"), format="y-m-d")
  } else if (tolower(Type)=="avyear"){
    StartDateOut <- dates(paste(Season-1, "10", "01", sep="-"), format="y-m-d")
    EndDateOut   <- dates(paste(Season, "09", "30", sep="-"), format="y-m-d")
  } else if (tolower(Type)=="custom") {
    if (is.na(StartDate) | is.na(EndDate)) {
      stop("Please specify parameters 'StartDate' and 'EndDate' properly.")
    } else {
      if(as.numeric(substr(StartDate, 1, 2))>6) {
        StartDateOut <- dates(paste(Season-1, StartDate, sep="-"), format="y-m-d")
      } else {
        StartDateOut <- dates(paste(Season, StartDate, sep="-"), format="y-m-d")
      }
      EndDateOut   <- dates(paste(Season, EndDate, sep="-"), format="y-m-d")
    }
  } else {
    stop("Please specify parameter 'Type' properly: 'AvYear', 'Operation' or 'Custom'.")
  }

  Output <- list(StartDate=StartDateOut,
                 EndDate=EndDateOut)

  return (Output)

}
