#' Extracts HST from profiles
#'
#' Extracts HST from profiles
#' @param Profiles Single profile or list of profiles
#'
#' @return Dataframe with HST information
#' @export 
#'
extractHST <- function(Profiles) {
  
  ## Base function
  extractHSTOneProfile <- function(Profile) {
    
    ## Extracting Layers
    Lyrs <- Profile$layers
    
    ## Copy of original layers
    LyrsOrig <- Lyrs
    
    ## Calculate layer thickness
    Lyrs$thick <- diff(c(0, Lyrs$height))
    
    ## Eliminate SH layer at the top
    if (Lyrs$grain_type[nrow(Lyrs)] == "SH") {
      Lyrs <- Lyrs[-nrow(Lyrs),]
    }
    
    ## Eliminate DH layer at the top
    if (Lyrs$grain_type[nrow(Lyrs)] == "DH" & Lyrs$thick[nrow(Lyrs)] <= 5) {
      Lyrs <- Lyrs[-nrow(Lyrs),]
    }
    
    ## Eliminate thin FC layer at the top
    if (Lyrs$grain_type[nrow(Lyrs)] == "FC" & Lyrs$thick[nrow(Lyrs)] <= 5) {
      Lyrs <- Lyrs[-nrow(Lyrs),]
    }
    
    ## Identify top non-PP and non-DF layer
    if (sum(Lyrs$grain_type != "PP" & Lyrs$grain_type != "DF")>0) {
      MaxNonHST <- max(which(Lyrs$grain_type != "PP" & Lyrs$grain_type != "DF"), na.rm = T)
    } else {
      MaxNonHST <- NA
    }
    
    ## Calculate parameters
    hst_bottom <- ifelse(is.na(Lyrs$height[MaxNonHST]), 0, Lyrs$height[MaxNonHST])
    hst_top <- max(LyrsOrig$height)
    
    ## Return
    return(data.frame(date = Profile$date,
                      vstation_id = Profile$vstation_id,
                      hst_bottom = hst_bottom,
                      hst_top = hst_top,
                      hst_thick = hst_top - hst_bottom))
    
  }
  
  ## Apply function to all profiles
  HSTList <- lapply(Profiles, FUN = extractHSTOneProfile)
  
  ## Turn list of data frames into singe dataframe
  HST <- data.table::rbindlist(HSTList)
  class(HST) <- "data.frame"
  
  ## Reformat
  HST$date <- as.Date(HST$date)
  
  ## Eliminate duplicates
  HST <- unique(HST)
  
  ## Eliminate dates with no HST
  HST <- HST[HST$hst_thick > 0,]
  
  ## Return result
  return(HST)
  
}