#' Get snowprofiles from database
#'
#' @param DBName database name
#' @param Sites a single or vector of vstation_ids to query
#' @param DateStart Start date of profile query. Default NA gives you all dates.
#' @param DateEnd End date of profile query. By default equal to DateStart.
#' @param TimeDiffUTC Difference to UTM in hours. Default is -8 hours for PST.
#' @param dem add spatial information
#' @param SizeCheck Switch for including check whether more than 100,000 are being requested. Default is TRUE.
#' @param Verbose Switch for printing SQL query. Default is FALSE.
#'
#' @return list of snowprofile objects
#' @export
#'
#' @examples
#' require(SarpSnowGeneral)
#' require(SarpSnowAnalysis)
#' 
#' DBName <- "test2018"
#' Site <- "HRDPS_518_700"
#' Date <- "2018-01-07"
#' 
#' slopes <- listSlopes(DBName, dem)
#' 
#' ## Profile timeline
#' pro.timeline <- getProfiles(DBName, Sites = Site)
#' pro.timeline <- getProfiles(DBName, Sites = Site, DateStart = "2018-01-01", DateEnd = "2018-02-01")
#' 
#' ## Profile map with all slopes
#' pro.map <- getProfiles(DBName, DateStart = Date)
#' 
#' ## Profile map with only flat slopes
#' pro.flat <- getProfiles(DBName, DateStart = Date, Sites = subset(slopes, angle == 0)$vstation_id)

getProfiles <- function(DBName, Sites = NA, DateStart = NA, DateEnd = DateStart, TimeDiffUTC = -8, dem = NULL, SizeCheck = 100000, Verbose = FALSE) {  
  
  ## Initialize  
  Response <- "y"
  
  ## Base queries
  DataQuery <- paste0("SELECT ", 
                      "date AS datetimeUTC, date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)  as datetime, ",
                      "DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) AS date, TIME(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) AS time, ",
                      "vstation, vstation_id, angle, aspect, ", 
                      "height, DATE(date_add(deposition_date, INTERVAL ", TimeDiffUTC, " HOUR)) AS deposition_date, ", 
                      "density, temperature, lwc, grain_size, grain_class, hardness, ssi ",
                      "FROM `profiles` ")
  
  CountQuery <- "Select count(date) as Count FROM `profiles`"
    
    
  ## Adds restrictions to queries
  if (!is.na(DateStart) | !is.na(Sites)) {
    
    DataQuery  <- paste0(DataQuery,  " WHERE ")
    CountQuery <- paste0(CountQuery, " WHERE ")
    
    ## Adding location filter
    if (!is.na(Sites)) {
      location_str <- paste0("'", Sites, "'", collapse = ", ")
      DataQuery  <- paste0(DataQuery,  " `vstation_id` IN (", location_str, ")")
      CountQuery <- paste0(CountQuery, " `vstation_id` IN (", location_str, ")")
    }
    
    ## Adding date filter
    if (!is.na(DateStart)) {
      if (!is.na(Sites)) {
        DataQuery  <- paste0(DataQuery,  " AND ")
        CountQuery <- paste0(CountQuery, " AND ")
      }
      DataQuery  <- paste0(DataQuery,  "DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) >= '", DateStart, "' AND DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) <= '", DateEnd, "' ")
      CountQuery <- paste0(CountQuery, "DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) >= '", DateStart, "' AND DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) <= '", DateEnd, "' ")
    }
    
  }
  
  ## Size check
  if (!is.na(SizeCheck)) {
    NumRec <- SarpSnowGeneral::getRecordsFromQuery(DBName, CountQuery)$Count
    if (NumRec > SizeCheck) {
      Response <- readline(prompt = paste0("You are requesting ", NumRec, " records. Do you want to proceed (y/n)?: "))
      Response <- tolower(substr(Response, 1, 1))
    }
  }
  
  ## Processing query
  if (Response == "y") {
    
    # Send to database and format
    rawData <- SarpSnowGeneral::getRecordsFromQuery(DBName, DataQuery, Verbose = Verbose)
    
    # Add dem spatial info
    if (!is.null(dem)) { rawData <- merge(rawData, dem, by = 'vstation') }
    
    # Split into list of dataframe based on unique date-station combos
    splitData <- split(rawData, with(rawData, interaction(vstation_id, date), drop = TRUE))
    
    # Convert them into snowprofile objects
    spros <- lapply(splitData, snowprofileMysql, hasDem = TRUE)
    
    return(spros)
    
  } else {
    
    return(NULL)
  }
  
}