#' Get weather timeseries from database
#'
#' @param DBName Database name
#' @param Sites A single or vector of vstation_ids to query
#' @param DateStart Start date of date interval. Default is NA
#' @param DateEnd End date of date interval. Default is DateStart. This means that if DateStart is specified only, only a single date will be downloaded.
#' @param DailyTime. Time for daily record. Default is '05:00:00'. This means that only the 5am record is downloaded for each day. To download the hourly records, set DailyTime = NA.
#' @param TimeDiffUTM. Difference to UTM in hours. Default is -8 hours for PST.
#' @param dem add spatial information
#' @param SizeCheck Switch for including check for large downloads. Default is 100,000. Set to NA to avoid check.
#' @param Verbose Switch for printing SQL query. Default is FALSE.
#'
#' @return dataframe of weather
#' @export
#'
#' @examples
#' require(SarpSnowGeneral)
#' require(SarpSnowAnalysis)
#' 
#' DBName <- "test2018"
#' Site <- "HRDPS_518_700"
#' Date <- "2018-01-07"
#' 
#' slopes <- listSlopes(DBName, dem)
#' 
#' ## Get all weather
#' wx <- getWeather(DBName)
#' 
#' ## Weather timeline
#' wx.timeline <- getWeather(DBName, Sites = Site)
#' wx.timeline <- getWeather(DBName, Sites = Site, DateStart = "2018-01-01", DateEnd = "2018-02-01")
#' 
#' ## Profile map with all slopes
#' wx.map <- getWeather(DBName, DateStart = Date)
#' 
#' ## Profile map with only flat slopes
#' wx.flat <- getWeather(DBName, DateStart = Date, Sites = subset(slopes, angle == 0)$vstation_id)

getWeather <- function(DBName, Sites = NA, DateStart = NA, DateEnd = DateStart, DailyTime = "08:00:00", TimeDiffUTC = -8, dem = NULL, SizeCheck = 100000, Verbose = FALSE) {  
  
  ## Initialize  
  Response <- "y"
  
  ## Base queries
  DataQuery <- paste0("SELECT ", 
                  "date AS datetimeUTC, date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)  as datetime, ",
                  "DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) AS date, TIME(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) AS time, ",
                  "vstation, vstation_id, angle, aspect, ", 
                  "hs, hn24, hn72, wind_transport24, hoar, ssi, z_ssi, tss, ski_pen, iswr, ilwr, ta, rh, vw, dw, psum, snow, rain ",
                  "FROM `timeseries`")
  
  CountQuery <- "Select count(date) as Count FROM `timeseries`"
  
  
  ## Adds restrictions to queries
  if (!is.na(DateStart) | !is.na(Sites) | !is.na(DailyTime)) {
    
    DataQuery  <- paste0(DataQuery,  " WHERE ")
    CountQuery <- paste0(CountQuery, " WHERE ")
    
    ## Adding location filter
    if (!is.na(Sites)) {
      location_str <- paste0("'", Sites, "'", collapse = ", ")
      DataQuery  <- paste0(DataQuery,  " `vstation_id` IN (", location_str, ")")
      CountQuery <- paste0(CountQuery, " `vstation_id` IN (", location_str, ")")
    }
    
    ## Adding date filter
    if (!is.na(DateStart)) {
      if (!is.na(Sites)) {
        DataQuery  <- paste0(DataQuery,  " AND ")
        CountQuery <- paste0(CountQuery, " AND ")
      }
      DataQuery  <- paste0(DataQuery,  "DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) >= '", DateStart, "' AND DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) <= '", DateEnd, "' ")
      CountQuery <- paste0(CountQuery, "DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) >= '", DateStart, "' AND DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) <= '", DateEnd, "' ")
    }
    
    ## Adding time filter
    if (!is.na(DailyTime)) {
      if (!is.na(Sites) | !is.na(DateStart)) {
        DataQuery  <- paste0(DataQuery,  " AND ")
        CountQuery <- paste0(CountQuery, " AND ")
      }
      DataQuery  <- paste0(DataQuery,  "TIME(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) ='", DailyTime, "'")
      CountQuery <- paste0(CountQuery, "TIME(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) ='", DailyTime, "'")
    }
    
  }

  ## Size check
  if (!is.na(SizeCheck)) {
    NumRec <- SarpSnowGeneral::getRecordsFromQuery(DBName, CountQuery)$Count
    if (NumRec > SizeCheck) {
      Response <- readline(prompt = paste0("You are requesting ", NumRec, " records. Do you want to proceed (y/n)?: "))
      Response <- tolower(substr(Response, 1, 1))
    }
  }
  
  ## Processing query
  if (Response == "y") {
    
    # Send to database and format
    wx <- SarpSnowGeneral::getRecordsFromQuery(DBName, DataQuery, Verbose = Verbose)
    wx <- formatWeather(wx)
  
    # Add dem spatial info
    if (!is.null(dem)) { wx <- addSpatial(wx, dem) } 
  
    return(wx)
    
  } else {
    
    return(NULL)
  }
  
}
