#' Get weather daily summary timeseries from database
#'
#' @param DBName Database name
#' @param Sites A single or vector of vstation_ids to query
#' @param DateStart Start date of date interval. Default is NA
#' @param DateEnd End date of date interval. Default is DateStart. This means that if DateStart is specified only, only a single date will be downloaded.
#' @param DailyTime. Time for daily record. Default is '08:00:00'. This means that daily summary is from 7am the previous day to 8am current day.
#' @param TimeDiffUTM. Difference to UTM in hours. Default is -8 hours for PST.
#' @param dem add spatial information
#' @param SizeCheck Switch for including check for large downloads. Default is 100,000. Set to NA to avoid check.
#' @param Verbose Switch for printing SQL query. Default is FALSE.
#'
#' @return dataframe of weather
#' @export
#'
#' @examples
#' require(SarpSnowGeneral)
#' require(SarpSnowAnalysis)
#' 
#' DB <- "GNP"
#' Site <- "VIR152521"
#' 
#' DateStart <- "2016-01-01"
#' DateEnd <- "2016-05-01"
#' 
#' ## Retrieve data 
#' WxDS <- getWeatherDailySummary("GNP", Sites = Site, DateStart = DateStart, DateEnd = DateEnd)
#' 
#' ## Base plot
#' ylim <- c(min(WxDS$ta_min), max(WxDS$ta_max))
#' plot(WxDS$date, WxDS$ta_avg, type = "n", ylim=ylim, ylab = "Temp", xlab = "Dates", main = "Daily Temperature Summary")
#' grid()
#' abline(h = 0, lwd = 2, lty = 2, col="grey")
#' 
#' ## Plotting
#' lines(WxDS$date, WxDS$ta_avg, col="black", lwd = 2)
#' lines(WxDS$date, WxDS$ta_min, col="blue")
#' lines(WxDS$date, WxDS$ta_max, col="red")

getWeatherDailySummary <- function(DBName, Sites, DateStart = NA, DateEnd = DateStart, DailyTime = "08:00:00", TimeDiffUTC = -8, dem = NULL, Verbose = FALSE) {  
  
  ## Base queries
  DataQuery <- paste0("SELECT ", 
                      "min(vstation_id) as vstation_id, ",
                      "date(date_add(date, INTERVAL ", TimeDiffUTC + 24 - as.numeric(substr(DailyTime, 1, 2)) - 1, " HOUR)) as `date`, ",
                      "time('", as.numeric(substr(DailyTime, 1, 2)), ":00:00') As `time`, ",
                      "count(id) as num_obs, ",
                      "round(avg(ta),2) as ta_avg, round(min(ta),2) as ta_min, round(max(ta),2) as ta_max ", 
                      "FROM `timeseries`")
  
  ## Adds restrictions to queries
  if (!is.na(DateStart) | !is.na(Sites) | !is.na(DailyTime)) {
    
    DataQuery  <- paste0(DataQuery,  " WHERE ")
    
    ## Adding location filter
    if (!is.na(Sites)) {
      location_str <- paste0("'", Sites, "'", collapse = ", ")
      DataQuery  <- paste0(DataQuery,  " `vstation_id` IN (", location_str, ")")
    }
    
    ## Adding date filter
    if (!is.na(DateStart)) {
      if (!is.na(Sites)) {
        DataQuery  <- paste0(DataQuery,  " AND ")
      }
      DataQuery  <- paste0(DataQuery,  "DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) >= '", DateStart, "' AND DATE(date_add(date, INTERVAL ", TimeDiffUTC, " HOUR)) <= '", DateEnd, "' ")
    }
    
  }
                      
  ## Grouping part of query
  DataQuery  <- paste0(DataQuery, " GROUP BY vstation_id, date(date_add(date, INTERVAL ", TimeDiffUTC + 24 - as.numeric(substr(DailyTime, 1, 2)) - 1, " HOUR))")                    

  wx <- SarpSnowGeneral::getRecordsFromQuery(DBName, DataQuery, Verbose = Verbose)
  wx <- formatWeatherDailySummary(wx)
  
  # Add dem spatial info
  if (!is.null(dem)) { wx <- addSpatial(wx, dem) } 
  
  return(wx)
    
}
