#' Merge snowprofiles
#' 
#' Take a list of snowprofile objects and return a single snowprofile object
#' @param x list of snowprofiles
#'
#' @return as single snowprofile
#' @export
#'

mergeSnowprofiles <- function(x) {
  
  # Handle case where x is a single profile, or not a list of profiles
  if (is(x, "snowprofiles")) {return(x)}
  if (!is(x, "list")) {return()} else {
    # Continue if x is a list of profiles
    
    # Make sure all elements are snowprofiles
    x <- x[sapply(x, class) == "snowprofile"]
    
    # Resample each profile to cover all possible deposition dates
    deposition.timestamps <- do.call('c', lapply(x, function(xx) xx$layers$deposition_date))
    deposition.timestamps <- sort(unique(as.Date(deposition.timestamps)))
    deposition.timestamps <- data.frame(deposition_date = deposition.timestamps)
    prf.norm <- lapply(x, function(xx) normalizeByDates(xx$layers, deposition.timestamps))
    
    # Merge properties
    layers <- deposition.timestamps
    layers$height <- rowMeans(do.call(cbind, lapply(prf.norm, "[", "height")), na.rm = T)
    layers$grain_size <- rowMeans(do.call(cbind, lapply(prf.norm, "[", "grain_size")), na.rm = T)
    layers$hardness <- rowMeans(do.call(cbind, lapply(prf.norm, "[", "hardness")), na.rm = T)
    layers$density <- rowMeans(do.call(cbind, lapply(prf.norm, "[", "density")), na.rm = T)
    layers$temperature <- rowMeans(do.call(cbind, lapply(prf.norm, "[", "temperature")), na.rm = T)
    layers$lwc <- rowMeans(do.call(cbind, lapply(prf.norm, "[", "lwc")), na.rm = T)
    layers$ssi <- rowMeans(do.call(cbind, lapply(prf.norm, "[", "ssi")), na.rm = T)
    
    # Most common grain type
    grain_type_table <- do.call(cbind, lapply(prf.norm, "[", "grain_type"))
    layers$grain_type <- apply(grain_type_table, 1, function(x) { tab <- table(x); names(tab)[which.max(tab)] } )
    
    # Add metadata
    out.profile <- list(layers = layers)
    out.profile$date <- unique(sapply(x, function(xx) xx$date))
    out.profile$vstation <- unique(sapply(x, function(xx) xx$vstation))
    out.profile$vstation_id <- unique(sapply(x, function(xx) xx$vstation_id))
    out.profile$aspect <- sapply(x, function(xx) xx$aspect)
    out.profile$angle <- sapply(x, function(xx) xx$angle)
    out.profile$elev <- sapply(x, function(xx) xx$elev)
    out.profile$profileType <- "aggregate"
    class(out.profile) <- "snowprofile"
    
    ## Return
    return(out.profile)}
}

