#' Normalize profiles by deposition date
#'
#' Resample profile over a list of deposition dates
#' @param profile snowprofile
#' @param prf.dates list of deposition dates
#'
#' @return A sparse snowprofile with interpolated values for all dates
#' @export
#'
normalizeByDates <- function(profile, prf.dates)
{
  #require(zoo) # interpolating over NA values (TODO: is this possible in base R?????)
  
  # Aggregate layers that have same depostion date
  profile <- aggregateByDeposition(profile)
  
  # Create a sparse profile that has a row for every possible deposition date
  sparse.profile <- merge(prf.dates, profile, all.x = T)
  
  # Apply a linear/spline interpolation between all dates for numeric columns
  numeric.cols <- names(sparse.profile) [sapply(sparse.profile, class) %in% c('numeric','logical')]
  sparse.profile[numeric.cols] <- zoo::na.approx(sparse.profile[numeric.cols], na.rm = F)
  
  # # Spline interpolation can create values outside realistic ranges
  # sparse.profile[numeric.cols] <- zoo::na.spline(sparse.profile[numeric.cols])
  
  return(sparse.profile)                               
}