#' Plot time series of snow profiles
#'
#' Plot time series of snow profiles
#' @param Profiles List of snow profiles created with getProfiles() function
#' @param Type TYpe of profile to be plotted. So far the following types are available: graintype (default), hardness, temperature, density, grainsize.
#' @param WithXAxis Switch for plotting x-axis (default = TRUE).
#' @param DateStart Start date for plot. If not provided, the function takes the date range from Profiles (default = NA).
#' @param DateEnd End date for plot. If not provided, the function takes the date range from Profiles (default = NA).
#' @param TrackingDates Date or array of dates for tracing layers (default = NA).
#' @param TrackingCol Color of tracking lines (default = "black")
#' @param TrackingLwd Width of tracking lines (default = 2)
#' @param HatchHST Switch for whether HST should be hatched (default is FALSE).
#' @param HighlightDateStart Start of highlighted time period (default = NA).
#' @param HighlightDateEnd End of highlighted time period (default = HighlightDateStart).
#' @param HighlightCol Color for highlighted time period (default = "light grey").
#' @param cex.axis cex.axis (default = 1.0).
#' @param cex.lab cex.lab (default = 1.0).
#' @param SiteName SiteName (default = NULL).
#'
#' @examples
#' require(SarpSnowAnalysis)
#' 
#' ## Get data
#' DBName <- "test2018"
#' Site <- "HRDPS_518_700"
#' Profiles <- getProfiles(DBName, Sites = Site)
#' 
#' ## Plot snowprofile
#' plotTSSnowProfile(Profiles, DateStart = "2017-12-01", DateEnd = "2018-04-01", 
#'                   HighlightDateStart = "2017-12-30", HighlightDateEnd = "2018-01-05",
#'                   TrackingDates = "2017-12-30", SiteName = Site, main = "Seasonal Snowprofile with 2018-01-05 layer tracked")
#' 
#' plotTSSnowProfile(Profiles, Type = "density", DateStart = "2017-12-01", DateEnd = "2018-04-01", 
#'                   HighlightDateStart = "2017-12-30", HighlightDateEnd = "2018-01-05",
#'                   TrackingDates = "2017-12-30", SiteName = Site, main = "Seasonal Snowprofile with 2018-01-05 layer tracked: DENSITY")
#' 
#' plotTSSnowProfile(Profiles, Type = "temp", DateStart = "2017-12-01", DateEnd = "2018-04-01", 
#'                   HighlightDateStart = "2017-12-30", HighlightDateEnd = "2018-01-05",
#'                   TrackingDates = "2017-12-30", SiteName = Site, main = "Seasonal Snowprofile with 2018-01-05 layer tracked: SNOW TEMPERATURE")
#' 
#' plotTSSnowProfile(Profiles, Type = "grainsize", DateStart = "2017-12-01", DateEnd = "2018-04-01", 
#'                   HighlightDateStart = "2017-12-30", HighlightDateEnd = "2018-01-05",
#'                   TrackingDates = "2017-12-30", SiteName = Site, main = "Seasonal Snowprofile with 2018-01-05 layer tracked: GRAIN SIZE")
#' 
#' plotTSSnowProfile(Profiles, Type = "hardness", DateStart = "2017-12-01", DateEnd = "2018-04-01", 
#'                   HighlightDateStart = "2017-12-30", HighlightDateEnd = "2018-01-05",
#'                   TrackingDates = "2017-12-30", SiteName = Site, main = "Seasonal Snowprofile with 2018-01-05 layer tracked: HARDNESS")
#'                   
#' plotTSSnowProfile(Profiles, Type = "ssi", DateStart = "2017-12-01", DateEnd = "2018-04-01", 
#'                   HighlightDateStart = "2017-12-30", HighlightDateEnd = "2018-01-05",
#'                   TrackingDates = "2017-12-30", SiteName = Site, main = "Seasonal Snowprofile with 2018-01-05 layer tracked: SSI")
#' 
#' @export

plotTSSnowProfile <- function(Profiles,
                              Type = "graintype",
                              WithXAxis = T,
                              DateStart = NA,
                              DateEnd = NA,
                              TrackingDates = NA,
                              TrackingCol = "black",
                              TrackingLwd = 2,
                              HatchHST = FALSE,
                              HighlightDateStart = NA,
                              HighlightDateEnd = HighlightDateStart,
                              HighlightCol = "light grey",
                              cex.axis = 1.0,
                              cex.lab = 1.0,
                              SiteName = NULL, 
                              main = NA) {
  
  print("... starting plotTSSnowprofile")
  
  ## Prep
  Type <- tolower(Type)
  
  ## Check
  if (Type != "graintype" & Type != "hardness" & Type != "density" & substr(Type, 1, 4) != "temp" & Type != "grainsize" & Type != "ssi") {
    stop(paste0("Profile type: '", Type, "' not supported!"))
  }
  
  ## Assign date range for chart
  if (is.na(DateStart)) {DateStart <- min(unlist(lapply(Profiles, function(x) substr(x$date, 1, 10))))}
  if (is.na(DateEnd))   {DateEnd   <- max(unlist(lapply(Profiles, function(x) substr(x$date, 1, 10))))}
  
  DateStart <- as.Date(DateStart)
  DateEnd <- as.Date(DateEnd)
  
  ## Calculate maximum HS
  HSmax <- max(unlist(lapply(Profiles, function(x) max(x$layers$height))))
  
  ## Set up plot
  plot(NA, NA, t = "n", xlim = c(DateStart, DateEnd), ylim = c(0, HSmax + 10), xaxt = "n", ylab = "HS", xlab = "", las = 1)
  addTimeGridAndHighlight(c(DateStart, DateEnd), WithXAxis = WithXAxis,
                          HighlightDateStart = HighlightDateStart, HighlightDateEnd = HighlightDateEnd,
                          HighlightCol = HighlightCol, cex.axis = cex.axis, cex.lab = cex.lab)
  
  
  
  ## Adding layers
  for (i in 1:length(Profiles)) {
    
    Lyrs <- Profiles[[i]]$layers
    Date <- as.Date(substr(Profiles[[i]]$date, 1, 10))
    
    if (tolower(Type) == "graintype") {
      Lyrs$Col <- grainColLookup(Lyrs$grain_type)
    } else if (tolower(Type) == "hardness") {
      Lyrs$Col <- getColoursHardness(Lyrs$hardness)
    } else if (tolower(Type) == "density") {
      Lyrs$Col <- getColoursDensity(Lyrs$density)
    } else if (substr(tolower(Type), 1, 4) == "temp") {
      Lyrs$Col <- getColoursSnowTemp(Lyrs$temperature)
    } else if (tolower(Type) == "grainsize") {
      Lyrs$Col <- getColoursGrainSize(Lyrs$grain_size)
    } else {
      Lyrs$Col <- "#E8E8E8"
    }
    
    for (l in 1:nrow(Lyrs)) {
      
      rect(xleft = Date - 0.5, 
           ybottom = ifelse(l == 1, 0, Lyrs$height[l - 1]), 
           xright = Date + 0.5,
           ytop = Lyrs$height[l], 
           border = NA,
           col = Lyrs$Col[l])
      
    }
    
    if (tolower(Type) == "ssi") {
      
      Lyrs$Col <- getColoursSSI(Lyrs$ssi)
      
      for (l in 1:(nrow(Lyrs) - 1)) {
        
        rect(xleft = Date - 0.5, 
             ybottom = Lyrs$height[l] - 0.5, 
             xright = Date + 0.5,
             ytop = Lyrs$height[l], 
             border = NA,
             col = Lyrs$Col[l])
        
      } 
      
    }
    
  }
  
  ## Hatch HST
  if (HatchHST) {
    HST <- extractHST(Profiles)
    rect(xleft = HST$date - 0.5, ybottom = HST$hst_bottom, xright = HST$date + 0.5, ytop = HST$hst_top, density = 25)
  }
  
  ## Adding layer tracking
  if (!is.na(TrackingDates[1])) {
    for (i in 1:length(TrackingDates)) {
      DatesArray <- as.Date(unlist(lapply(Profiles, function(x) substr(x$date, 1, 10))))
      DepthArray <- unlist(lapply(Profiles, function(x) x$layers$height[max(which(x$layers$deposition_date <= as.Date(TrackingDates[i])))]))
      lines.default(DatesArray[DatesArray >= as.Date(TrackingDates[i])], DepthArray[DatesArray >= as.Date(TrackingDates[i])], lwd = TrackingLwd, col = TrackingCol)
    }
  }
  
  ## Adding grid lines on top
  DateArray <- seq(from = DateStart, to = DateEnd, by = "days")
  Saturdays <- DateArray[weekdays(DateArray) == "Saturday"]
  abline(v = Saturdays, lty = 3, col = "dark grey")
  grid(nx = NA, ny = NULL, lty = 3, col = "dark grey")
  box()
  
  ## Adding title
  if (!is.na(main)) {
    title(main)
  }
  
  ## Adding station name
  if (!is.null(SiteName)) {
    mtext(text = SiteName, side = 3, line = 0.2, adj = 1, cex = cex.axis*0.5)
  }
  
}
