#' Plot time series of weather and snowpack parameters from snowpack simulations
#'
#' Plot time series of weather and snowpack parameters from snowpack simulations
#' @param Wx Weather dataframe created with getWeather() function
#' @param ParamName Name of parameter to be plotted
#' @param WithXAxis Switch for plotting x-axis (default = TRUE).
#' @param DateStart Start date for plot. If not provided, the function takes the date range from Wx (default = NA).
#' @param DateEnd End date for plot. If not provided, the function takes the date range from Wx (default = NA).
#' @param HighlightDateStart Start of highlighted time period (default = NA).
#' @param HighlightDateEnd End of highlighted time period (default = HighlightDateStart).
#' @param HighlightCol Color for highlighted time period (default = "light grey").
#' @param cex.axis cex.axis (default = 1.0).
#' @param cex.lab cex.lab (default = 1.0).
#' @param SiteName SiteName (default = NULL).
#'
#' @examples
#' require(SarpSnowAnalysis)
#' 
#' ## Get data
#' DBName <- "test2018"
#' Site <- "HRDPS_518_700"
#' Wx <- getWeather(DBName, Sites = Site)
#' 
#' ## Plot snowprofile
#' plotTSSnowpackWx(Wx, DateStart = "2017-12-01", DateEnd = "2018-04-01", 
#'                  HighlightDateStart = "2017-12-30", HighlightDateEnd = "2018-01-05",
#'                  TrackingDates = "2017-12-30", SiteName = Site, main = "Seasonal Snowprofile with 2018-01-05 layer tracked")
#'
#' @export

plotTSSnowpackWx <- function(Wx,
                             ParamName,
                             WithXAxis = T,
                             DateStart = NA,
                             DateEnd = NA,
                             FaintLines = TRUE,
                             FaintCol = "#A9A9A9",
                             FaintLwd = 0.25,
                             SummaryQuantiles = c(0.5, 0.25, 0.75),
                             Threshold = NA,
                             ThresholdCol = "#B22222",
                             HighlightDateStart = NA,
                             HighlightDateEnd = HighlightDateStart,
                             HighlightCol = "light grey",
                             cex.axis = 1.0,
                             cex.lab = 1.0,
                             SiteName = NULL, 
                             ylim = NA,
                             main = NA) {
  
  print("... starting plotTSSnowpackWx")
  
  ## Preparation
  ParamName <- tolower(ParamName)
  
  ## Create reduced df with plotting variables
  PlotDF <- data.frame(vstation_id = Wx$vstation_id,
                       date = Wx$date,
                       target = Wx[,ParamName])
  
  ## Converting df into list of df
  PlotDFList <- split(PlotDF, PlotDF$vstation_id, drop = TRUE)
  
  ## Calculate quantiles
  Summary <- aggregate(PlotDF$target, by = list(PlotDF$date), FUN = quantile, na.rm = T, probs = SummaryQuantiles)
  Summary <- cbind(as.Date(Summary[,1]), as.data.frame(Summary$x))
  names(Summary)[1] <- "date"

  ## Determine dates that are above the threshold
  if (!is.na(Threshold)) {
    Summary$Thresh <- Threshold
    Summary$AboveThresh <- ifelse(Summary[,2] >= Threshold, 1, 0)
  }

  # assign date range for chart
  if (is.na(DateStart)) {DateStart <- min(PlotDF$date)}
  if (is.na(DateEnd))   {DateEnd   <- max(PlotDF$date)}
  
  DateStart <- as.Date(DateStart)
  DateEnd <- as.Date(DateEnd)
  
  ## Calculate range for y-axis
  if (is.na(ylim[1])) {
    ylim <- c(min(PlotDF$target, na.rm = T), max(PlotDF$target, na.rm = T))
  }
  
  ## Set up plot
  plot(NA, NA, t = "n", xlim = c(DateStart, DateEnd), ylim = ylim, xaxt = "n", ylab = ParamName, xlab = "", las = 1)
  addTimeGridAndHighlight(c(DateStart, DateEnd), WithXAxis = WithXAxis,
                          HighlightDateStart = HighlightDateStart, HighlightDateEnd = HighlightDateEnd,
                          HighlightCol = HighlightCol, cex.axis = cex.axis, cex.lab = cex.lab)
  
  ## Add above threshold markings
  if (!is.na(Threshold)) {
    Above <- Summary[Summary$Above == 1,]
    for (i in 1:nrow(Above)) {
      rect(xleft = Above$date[i] - 0.5, xright = Above$date[i] + 0.5, ybottom = -1000, ytop = 1000, col = paste0(ThresholdCol, "30"), border = NA)
    }
  }
  
  
  ## Adding faint individual lines
  if (FaintLines) {
    invisible(lapply(PlotDFList, function(x) lines(x$date, x$target, lwd = FaintLwd, col = paste0(FaintCol, "50"))))
  }
  
  
  ## Adding solid lines
  if (length(PlotDFList) == 1) {
    
    lines(PlotDFList[[1]]$date, PlotDFList[[1]]$target, lwd = 1)
    
  } else {
    
    lines(Summary$date, Summary[,2], lwd = 1, lty = 1)
    
    if (!is.na(Threshold) & ncol(Summary) > 4) {
      for (i in 3:(ncol(Summary) - 2)) {
        lines(Summary$date, Summary[,i], lwd = 1, lty = 2)
      }
    } else if (is.na(Threshold) & ncol(Summary) > 2) {
      for (i in 3:ncol(Summary)) {
        lines(Summary$date, Summary[,i], lwd = 1, lty = 2)
      }
    }
    
  }
  
  
  ## Adding grid lines on top
  DateArray <- seq(from = DateStart, to = DateEnd, by = "days")
  Saturdays <- DateArray[weekdays(DateArray) == "Saturday"]
  abline(v = Saturdays, lty = 3, col = "dark grey")
  grid(nx = NA, ny = NULL, lty = 3, col = "dark grey")
  box()
  
  ## Adding threshold line
  if (!is.na(Threshold)) {
    abline(h = Threshold, lwd = 2, lty = 2, col = ThresholdCol)
  }
  
  ## Adding title
  if (!is.na(main)) {
    title(main)
  }
  
  ## Adding station name
  if (!is.null(SiteName)) {
    mtext(text = SiteName, side = 3, line = 0.2, adj = 1, cex = cex.axis*0.5)
  }
  
  if (length(PlotDFList) > 1) {
    text <- paste0("Number of sites: ", length(PlotDFList))
    if (!is.na(SummaryQuantiles[1])) {
      text <- paste0(text, " (Quantiles: ", paste(SummaryQuantiles, collapse = ", "), ")")
    }
    mtext(text = text, side = 3, line = 0.2, adj = 0, cex = cex.axis*0.5)
  }
  
  ## Return 
  invisible(Summary)
  
}
