#' Gets colors for plotting SSI values
#'
#' Gets colors for plotting SSI values in snowprofiles.
#' @param Values SSI values
#' @param Resolution Resolution of color scale. Default is 100.
#' @param Verbose Switch for writing out value and html color tuplets for debugging. 
#' @return Array with HTML color codes
#' 
#' @examples
#' require(SarpSnowGeneral)
#' 
#' SSI <- c(-999, 1:6)
#' plot(rep(1,length(SSI)), bg = getColoursSSI(SSI), pch = 21,cex = 3)
#' text(x = c(1:length(SSI)), y = 1, labels = SSI, pos = 3, offset = 2)
#' 
#' @export

getColoursSSI <- function(Values, Resolution = 101, Verbose = F) {
  
  ## Base function 
  getColourSSI <- function(Value, Resolution, Verbose = F) {
    
    ## Colors: Color brewer yellow to red (6 levels)
    ## 1: #bd0026
    ## 2: #f03b20
    ## 3: #fd8d3c
    ## 4: #feb24c
    ## 5: #fed976
    ## 6: #ffffb2
    
    ClrPalette <- colorRampPalette(c("#bd0026", "#f03b20", "#fd8d3c", "#feb24c", "#fed976", "#ffffb2"))
    ClrRamp <- ClrPalette(Resolution)
    
    if (is.na(Value)) {
      Clr <- "#FFFFFF"
    } else if (Value == -999) {
      Clr <- "#F0F0F0"
    } else if (Value > 6 | Value < 1) {
      Clr <- "#FFFFFF"
    } else {
      Clr <- ClrRamp[1 + round((Value - 1)/5 * (Resolution - 1), 0)]
    }
    
    if (Verbose) {print(paste(Value, "-", Clr))}
    return(Clr)
    
  }
  
  ## Application to array
  Clrs <- unlist(sapply(Values, getColourSSI, Resolution = Resolution, Verbose = Verbose))
  
  return(Clrs)
  
}