#' Plot map of study area 
#'
#' Map with outline of study area and NWP grid points
#' @param studyArea the studyArea object (output of a calcStudyArea function)
#' @param nwpModel "HRDPS" (default) or "HRDPS_WEST"
#' @param colCode "band" (default) uses 3 colours for BTL/TL/ALP, "elevation" is 6 levels of colours based on elevation, "zone" is coloured by subregion
#' @param minElevation only show elevations above threshold
#'
#' @export
#'
#' @examples
#' gnp <- readRDS(url("http://data.avalancheresearch.ca/Snowmodels/studyAreas/gnp.rds"))
#' par(mfrow = c(1,2))
#' plot(gnp)
#' plot(gnp, "HRDPS_WEST")
#' plot(gnp, Google = TRUE, cex = 3)
#' 

plot.studyArea <- function(studyArea, nwpModel = 'HRDPS', Google = FALSE, colCode = 'band', minElevation = NULL, ...)
{
  
  # Select nwpModel to plot
  dem <- studyArea[[nwpModel]]
  # Only show high elevations
  if (!is.null(minElevation)) {dem <- subset(dem, elev > minElevation)}
  
  # Plot study area with white background
  if (!Google)
  {
    plot(studyArea$geometry)
    
    # Plot study area on Google basemap  
  } else {
    
    # Load a basemap with dimensions of DEM
    bbox <- RgoogleMaps::qbbox(lat = dem@coords[,2], lon = dem@coords[,1])
    bgmap <- RgoogleMaps::GetMap.bbox(bbox$lonR, bbox$latR, 
                                      maptype = 'terrain', 
                                      frame = F, 
                                      destfile = file.path(tempdir(), 'MyTile.png.rda'))
    
    # Plot study area polygon
    RgoogleMaps::PlotPolysOnStaticMap(bgmap,
                                      sp::SpatialPolygons(studyArea$geometry@polygons), 
                                      add = FALSE, mar = c(3,3,3,2))
    
    # Reproject points onto GoogleMaps tile dimensions
    newCoords <- data.frame(RgoogleMaps::LatLon2XY.centered(bgmap, lat = dem@coords[,2], lon = dem@coords[,1]))
    dem@coords[,1] <- newCoords[,1]
    dem@coords[,2] <- newCoords[,2]
  }
  
  # Colour options
  if (colCode == 'band') {
    mycols <- getBandColours(dem$band)
    
  } else if (colCode == 'zone') {
    mycols <- rainbow(length(unique(dem$zone)))[factor(dem$zone)]
    
  } else if (colCode == 'elevation') {
    palette(RColorBrewer::brewer.pal(8, "YlGnBu"))
    mycols <- plotrix::rescale(dem$elev, c(1, 8))
  }
  
  # Draw points
  points(dem, pch = 20, col = mycols, ...)
  title(nwpModel)
  
  # Reset palette
  palette("default")
  
}