# Create studyArea class
studyArea <- function(x) UseMethod("studyArea", x)

# Check if object is a studyArea
is.studyArea <- function(x) inherits(x, "studyArea")


#' Default constructor for a studyArea object
#'
#' @param name name of the study area
#' @param geometry predefined geometry of the study area (SpatialPolygons)
#' @param hrdps DEM of HRDPS model
#' @param hrdpswest DEM of HRDPS_WEST model
#' @param elevationBands DEM contains preset treeline elevations for each gridpoint
#' @param name.col column name from the geometry that contains the names of subregions (zones)
#'
#' @return studyArea object
#' @export
#'
#' @examples
#' See example for specific constructor (kml/bulletin/gps/points)
#' 
studyArea.default <- function(name, geometry, hrdps = NULL, hrdpswest = NULL, elevationBands = TRUE, name.col = 'name')
{
  object <- list()
  object$name <- name 
  
  # Add geometry
  geometry <- sp::spTransform(geometry, sp::CRS('+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs'))
  object$geometry <- geometry
  
  # Load appropraite function based on geometry
  if (class(geometry) == "SpatialPolygonsDataFrame") {findPoints <- findPointsInPolygons}
  if (class(geometry) == "SpatialPointsDataFrame")   {findPoints <- findNearestPoints}
  
  # Add NWP grid points
  print("Finding grid points in region")
  # HRPDS
  if (!is.null(hrdps)) {
    object$HRDPS <- findPoints(object$geometry, hrdps, name.col)
    if (elevationBands) {object$HRDPS <- setElevationBands(object$HRDPS, treeline.widths = c(200, 0))}
  }
  # HRDPS_WEST
  if (!is.null(hrdpswest)) {
    object$HRDPS_WEST <- findPoints(object$geometry, hrdpswest, name.col)
    if (elevationBands) {object$HRDPS_WEST <- setElevationBands(object$HRDPS_WEST, treeline.widths = c(200, 0))}
  }
  
  # Return studyArea object
  class(object) <- "studyArea"
  return(object)
}
