#' Determine study area for an InfoEx operation with GPS tracking data
#' 
#' @param name name of the study area
#' @param operationCode name of the operation in the database
#' @param hrdps DEM of HRDPS model
#' @param hrdpswest DEM of HRDPS_WEST model
#' @param elevationBands DEM contains preset treeline elevations for each gridpoint
#'
#' @return studyArea object
#' @export
#'
#' @examples
#' library(SarpGPSTools)
#' library(SarpGPSToolsPrivate)
#' hrdps <- readRDS(url("http://data.avalancheresearch.ca/Snowmodels/dem/HRDPS_DEM.rds"))
#' hrdpswest <- readRDS(url("http://data.avalancheresearch.ca/Snowmodels/dem/HRDPS_WEST_DEM.rds"))
#' whs <- studyAreaGps("WHS", "WHS", hrdps, hrdpswest)
#'

studyAreaGps <- function(name, operationCode, hrdps = NULL, hrdpswest = NULL, elevationBands = TRUE)
{
  
  # Extract geometry of operation
  print("Extracting geometry of operation")
  geometry <- SarpGPSTools::getLocCatlog(operationCode)
  
  # Some catalogues have messy zones
  geometry <- raster::crop(geometry$Zones, extent(geometry$OpArea))
  
  # Create study area
  studyArea <- studyArea.default(name, geometry, hrdps, hrdpswest, elevationBands)
  
  return(studyArea)
}